#!/usr/bin/wish8.6
set version 0.5

# "Deconv2Dxy" is useful for lineshape deconvolution for 1D- and 2D-NMR spectra.
#
#    Copyright (C) 2018, Jörn Schmedt auf der Günne
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.


#author: Joern Schmedt auf der Guenne
#13.08.2018
#
########know-how########
#-lineshape function:
# lineshape = ar*gl*2/wi*sqrt(ln(2)/pi)*exp(-4*ln(2)/wi/wi*(x-x0)^2)+
#            ar*(1-gl)*0.5/pi*wi/(wi^2/4+(x-x0)^2)
#-second moment for pure Gaussian functions (gl=1) is calculated; 2nd_moment*sqrt(8*ln(2))=FWHM
#-load lineshape activates all variables for fitting (if possible) with the exception of the Gauss/Lorentzian value
#-wi = FWHM both for Gaussian and Lorentzian

# Note:

#canvas size
set canvaswidthx [expr [winfo screenwidth .]-300]
set canvaswidthy [expr [winfo screenheight .]-120]
#points of graphical resolution
set pores 891
#verbosity while fitting
set verbose 1
#set nlb 10
set iterationsperpeak 6000
#fit cutoff:
#cut happens approx at fact*znoise
set fact 0.05
#xi2 convergence
set epsilon 1.0e-14

#don't change anything below
set npeaks 0
set currentpeak(peak) 0
set pi 3.14159265358979323846264338
#global variables
#zmatrix 
#zline
#xline
#phase,boolfit
set ppmnothertz 1
set reffreq 1.0e6
set refoffset 0.0
set phase 0.0
array set boolfit [list phase 0 shift 1 area 1 gl 1 width 1]

#note: area is chosen such that it equals the area of a peak
package require BLT 2.5

#-make temporary changes to tkfbox.tcl to suppress hiddendirectories
#    set ::tk::dialog::file::showHiddenBtn 1
#    set ::tk::dialog::file::showHiddenVar 0
catch {tk_getOpenFile foo bar}
set ::tk::dialog::file::showHiddenBtn 1
set ::tk::dialog::file::showHiddenVar 0

# PROC print
# Description:
proc printplot {filename} {
#  lappend ::auto_path /usr/lib/tcltk/Img1.3
  set null [catch {package require Img}]
  if {[catch {set image [image create photo -format window -data .rf.mg]}]} {
    set filetypes {
      {{postscript} {.ps}    }
    }
  } else {
    set filetypes {
      {{postscript} {.ps}    }
      {{bmp}        {.bmp}   }
      {{png}        {.png}   }
      {{jpg}        {.jpg}   }
      {{gif}        {.gif}   }
    }
    set image [image create photo -format window -data .rf.mg]
  }
  set filename [tk_getSaveFile -initialfile $filename.ps -title "plot" -filetypes $filetypes -defaultextension "postscript"]
  set extension [file extension $filename]
  if {$filename == ""} {
  } elseif {$extension==".ps"} {
    .rf.mg postscript output $filename -maxpect yes -decorations no
  } elseif {$extension==".bmp"} {
    $image write -format bmp "$filename"
    image delete $image
  } elseif {$extension==".png"} {
    $image write -format png "$filename"
    image delete $image
  } elseif {$extension==".jpg"} {
    $image write -format jpeg "$filename"
    image delete $image
  } elseif {$extension==".gif"} {
    $image write -format gif "$filename"
    image delete $image
  }
}

# Author of the original package "complex": Richard Suchenwirth
# http://wiki.tcl.tk/1612
 namespace eval complex {
    proc scan s {
#        regexp {^(-?([0-9]*\.)?[0-9]+)?(([+-]([0-9]*\.)?[0-9]*)i)?$} $s -> re - - im
        regexp {^(-?([0-9]*\.)?[0-9]+(e[-+]?[0-9]*)?)?(([+-]([0-9]*\.)?[0-9]*(e[-+]?[0-9]*)?)i)?$} $s -> re - - - im
	if {[expr ![info exists re]]} {
	 puts "\n\nincorrect format for complex numbers use:\n1.23+4.56i\n2-i\n3\n"
	 puts "You used: $s"
	 exit 0
        } elseif {$re==""} {
	  set re 0
	}
        switch -- $im {
            "" {set im 0}
            +  {set im 1}
            -  {set im -1}
        }
        list $re [expr {$im}] ;# expr may strip a plus sign
    }
    proc format {re {im 0}} {
        if {!$im} {return $re}
        subst $re[signof $im][expr {abs($im)==1?"":abs($im)}]i
    }
    proc signof x {expr {$x<0?"-":"+"}}
    proc re  x {lindex [scan $x] 0}
    proc im  x {lindex [scan $x] 1}
    proc abs x {expr hypot([join [scan $x] ,])} ;# no bracing with join
    proc arg x {expr atan2([im [format $x]],[re [format $x]])} ;# no bracing with join
    proc euler x {format [expr cos($x)] [expr sin($x)]} ;# no bracing with join
    proc + {x y} {
        foreach {a b} [scan $x] {c d} [scan $y] break
        format [expr {$a+$c}] [expr {$b+$d}]
    }
    proc - {x y} {
        foreach {a b} [scan $x] {c d} [scan $y] break
        format [expr {$a-$c}] [expr {$b-$d}]
    }
    proc * {x y} {
        foreach {a b} [scan $x] {c d} [scan $y] break
        format [expr {$a*$c-$b*$d}] [expr {$a*$d+$b*$c}]
    }
    proc / {x y} {
        foreach {a b} [scan $x] {c d} [scan $y] break
        set div [expr {double($c*$c+$d*$d)}]
        format [expr {($a*$c+$b*$d)/$div}] [expr {($b*$c-$a*$d)/$div}]
    }
 }


# PROC readrawfile name_of_raw_file
# Description:
proc readrawfile {name_of_raw_file} {
  if {[file exists $name_of_raw_file]} {
    if [catch {open $name_of_raw_file RDONLY} f] {
    } else {
      while {1} {
        gets $f zeile
        if [eof $f] break
        lappend field_of_ref $zeile
      }
      close $f
    } 
  } else {
    puts "File $name_of_raw_file does not exist!"
    set field_of_ref ""
  }
  return $field_of_ref
}

# PROC string2file (String, filename, Bool)
# Description:
# writes a string into a file. If file exists, it will overwrite the existing 
# file. It is possible to append the string to the filename by setting the 
# option to True
proc string2file {string nameoffile {boolappend 0}} {
  if {1!=$boolappend} {
    file delete -force  $nameoffile
  }
  set channel [open $nameoffile a+] 
  puts  $channel $string
  eof   $channel
  close $channel
}

# the "nelderMead" procedure has been taken from the tcllib and slightly modified, so that
# if the minimum is in the starting vertex it will remain in the simplex.
#----------------------------------------------------------------------
#
# nelderMead --
#
#	Attempt to minimize/maximize a function using the downhill
#	simplex method of Nelder and Mead.
#
# Usage:
#	nelderMead f x ?-keyword value?
#
# Parameters:
#	f - The function to minimize.  The function must be an incomplete
#	    Tcl command, to which will be appended N parameters.
#	x - The starting guess for the minimum; a vector of N parameters
#	    to be passed to the function f.
#
# Options:
#	-scale xscale
#		Initial guess as to the problem scale.  If '-scale' is
#		supplied, then the parameters will be varied by the
#	        specified amounts.  The '-scale' parameter must of the
#		same dimension as the 'x' vector, and all elements must
#		be nonzero.  Default is 0.0001 times the 'x' vector,
#		or 0.0001 for zero elements in the 'x' vector.
#
#	-epsilon epsilon
#		Requested tolerance in the function value; nelderMead
#		returns if N+1 consecutive iterates all differ by less
#		than the -epsilon value.  Default is 1.0e-7
#
#	-maxiter N
#		Maximum number of iterations to attempt.  Default is
#		500.
#
#	-trace flag
#		If '-trace 1' is supplied, nelderMead writes a record
#		of function evaluations to the standard output as it
#		goes.  Default is 0.
#
#----------------------------------------------------------------------

proc nelderMead { f startx args } {
  global epsilon
    array set params {
	-epsilon $epsilon
	-maxiter 500
	-scale {}
	-trace 0
    }
    # Check arguments
    if { ( [llength $args] % 2 ) != 0 } {
        return -code error -errorcode [list nelderMead wrongNumArgs] \
            "wrong \# args, should be\
                 \"[lreplace [info level 0] 1 end \
                         f x1 x2 ?-option value?...]\""
    }
    foreach { key value } $args {
        if { ![info exists params($key)] } {
            return -code error -errorcode [list nelderMead badoption $key] \
                "unknown option \"$key\",\
                     should be -epsilon, -maxiter, -scale or -trace"
        }
        set params($key) $value
    }
    # Construct the initial simplex
    set vertices [list $startx]
    if { [llength $params(-scale)] == 0 } {
	set i 0
	foreach x0 $startx {
	    if { $x0 == 0 } {
		set x1 0.0001
	    } else {
		set x1 [expr {1.0001 * $x0}]
	    }
	    lappend vertices [lreplace $startx $i $i $x1]
	    incr i
	}
    } elseif { [llength $params(-scale)] != [llength $startx] } {
	return -code error -errorcode [list nelderMead badOption -scale] \
	    "-scale vector must be of same size as starting x vector"
    } else {
	set i 0
	foreach x0 $startx s $params(-scale) {
	    lappend vertices [lreplace $startx $i $i [expr { $x0 + $s }]]
	    incr i
	}
    }
    # Evaluate at the initial points
    set n [llength $startx]
    foreach x $vertices {
	set cmd $f
	lappend cmd $x
	set y [uplevel 1 $cmd]
	if {$params(-trace)} {
	    puts "nelderMead: evaluating initial point: x=[list $x] y=$y"
	}
	lappend yvec $y
    }
    # Loop adjusting the simplex in the 'vertices' array.
    set nIter 0
    while { 1 } {
	# Find the highest, next highest, and lowest value in y,
	# and save the indices.
	set iBot 0
	set yBot [lindex $yvec 0]
	set iTop -1
	set yTop [lindex $yvec 0]
	set iNext -1
	set i 0
	foreach y $yvec {
	    if { $y <= $yBot } {
		set yBot $y
		set iBot $i
	    }
	    if { $iTop < 0 || $y >= $yTop } {
		set iNext $iTop
		set yNext $yTop
		set iTop $i
		set yTop $y
	    } elseif { $iNext < 0 || $y >= $yNext } {
		set iNext $i
		set yNext $y
	    }
	    incr i
	}
	# Return if the relative error is within an acceptable range
	set rerror [expr { 2. * abs( $yTop - $yBot )
			   / ( abs( $yTop ) + abs( $yBot ) ) }]
	if { $rerror < $params(-epsilon) } {
	    set status ok
	    break
	}
	# Count iterations
	if { [incr nIter] > $params(-maxiter) } {
	    set status too-many-iterations
	    break
	}
	incr nIter
	# Find the centroid of the face opposite the vertex that
	# maximizes the function value.
	set centroid {}
	for { set i 0 } { $i < $n } { incr i } {
	    lappend centroid 0.0
	}
	set i 0
	foreach v $vertices {
	    if { $i != $iTop } {
		set newCentroid {}
		foreach x0 $centroid x1 $v {
		    lappend newCentroid [expr { $x0 + $x1 }]
		}
		set centroid $newCentroid
	    }
	    incr i
	}
	set newCentroid {}
	foreach x $centroid {
	    lappend newCentroid [expr { $x / $n }]
	}
	set centroid $newCentroid
	# The first trial point is a reflection of the high point
	# around the centroid
	set trial {}
	foreach x0 [lindex $vertices $iTop] x1 $centroid {
	    lappend trial [expr {$x1 + ($x1 - $x0)}]
	}
	set cmd $f
	lappend cmd $trial
	set yTrial [uplevel 1 $cmd]
	if { $params(-trace) } {
	    puts "nelderMead: trying reflection: x=[list $trial] y=$yTrial"
	}
	# If that reflection yields a new minimum, replace the high point,
	# and additionally try dilating in the same direction.
	if { $yTrial < $yBot } {
	    set trial2 {}
	    foreach x0 $centroid x1 $trial {
		lappend trial2 [expr { $x1 + ($x1 - $x0) }]
	    }
	    set cmd $f
	    lappend cmd $trial2
	    set yTrial2 [uplevel 1 $cmd]
	    if { $params(-trace) } {
		puts "nelderMead: trying dilated reflection:\
                      x=[list $trial2] y=$y"
	    }
	    if { $yTrial2 < $yBot } {
		# Additional dilation yields a new minimum
		lset vertices $iTop $trial2
		lset yvec $iTop $yTrial2
	    } else {
		# Additional dilation failed, but we can still use
		# the first trial point.
		lset vertices $iTop $trial
		lset yvec $iTop $yTrial
	    }
	} elseif { $yTrial < $yNext } {
	    # The reflected point isn't a new minimum, but it's
	    # better than the second-highest.  Replace the old high
	    # point and try again.
	    lset vertices $iTop $trial
	    lset yvec $iTop $yTrial
	} else {
	    # The reflected point is worse than the second-highest point.
	    # If it's better than the highest, keep it... but in any case,
	    # we want to try contracting the simplex, because a further
	    # reflection will simply bring us back to the starting point.
	    if { $yTrial < $yTop } {
		lset vertices $iTop $trial
		lset yvec $iTop $yTrial
		set yTop $yTrial
	    }
	    set trial {}
	    foreach x0 [lindex $vertices $iTop] x1 $centroid {
		lappend trial [expr { ( $x0 + $x1 ) / 2. }]
	    }
	    set cmd $f
	    lappend cmd $trial
	    set yTrial [uplevel 1 $cmd]
	    if { $params(-trace) } {
		puts "nelderMead: contracting from high point:\
                      x=[list $trial] y=$y"
	    }
	    if { $yTrial < $yTop } {
		# Contraction gave an improvement, so continue with
		# the smaller simplex
		lset vertices $iTop $trial
		lset yvec $iTop $yTrial
	    } else {
		# Contraction gave no improvement either; we seem to
		# be in a valley of peculiar topology.  Contract the
		# simplex about the low point and try again.
		set newVertices {}
		set newYvec {}
		set i 0
		foreach v $vertices y $yvec {
		    if { $i == $iBot } {
			lappend newVertices $v
			lappend newYvec $y
		    } else {
			set newv {}
			foreach x0 $v x1 [lindex $vertices $iBot] {
			    lappend newv [expr { ($x0 + $x1) / 2. }]
			}
			lappend newVertices $newv
			set cmd $f
			lappend cmd $newv
			lappend newYvec [uplevel 1 $cmd]
			if { $params(-trace) } {
			    puts "nelderMead: contracting about low point:\
                                  x=[list $newv] y=$y"
			}
		    }
		    incr i
		}
		set vertices $newVertices
		set yvec $newYvec
	    }
	}
    }
    return [list y $yBot x [lindex $vertices $iBot] vertices $vertices yvec $yvec nIter $nIter status $status]
}

proc grad {f arg epsilon} {
  global fitpar
puts "fitparam: $fitpar"
puts "fitfunction $f"
puts "epsilon: $epsilon"
  set n [llength $arg]
  set vector ""
  set i 0
  foreach param $arg fitp $fitpar {
    if {[string match gl* $fitp] && $param <= $epsilon} {
puts case1
      set x1 [lreplace $arg $i $i [expr {$param}]]
      set x2 [lreplace $arg $i $i [expr {$param+$epsilon}]]
      set cmd $f
      lappend cmd $x1
      set f1 [uplevel 1 $cmd]
      set cmd $f
      lappend cmd $x2
      set f2 [uplevel 1 $cmd]
      set quot [expr {($f2-$f1)/$epsilon}]
      if {$quot > 0} {
        lappend vector 0.0
      } else {
        lappend vector $quot
      }
      incr i
    } elseif {[string match gl* $fitp] && $param >= 1.0-sqrt($epsilon)} {
puts case2
      set x1 [lreplace $arg $i $i [expr {$param-$epsilon}]]
      set x2 [lreplace $arg $i $i [expr {$param}]]
      set cmd $f
      lappend cmd $x1
      set f1 [uplevel 1 $cmd]
      set cmd $f
      lappend cmd $x2
      set f2 [uplevel 1 $cmd]
      lappend vector [expr {($f2-$f1)/$epsilon}]
      incr i
    } else {
puts case3
      set x1 [lreplace $arg $i $i [expr {$param-$epsilon}]]
      set x2 [lreplace $arg $i $i [expr {$param+$epsilon}]]
      set cmd $f
      lappend cmd $x1
      set f1 [uplevel 1 $cmd]
      set cmd $f
      lappend cmd $x2
      set f2 [uplevel 1 $cmd]
      lappend vector [expr {($f2-$f1)/(2.0*$epsilon)}]
      incr i
    }
  }
  return $vector
}

proc hesse {f arg epsilon} {
  global fitpar
  set n [llength $arg]
  set matrix ""
  set i 0
  foreach paramX $arg fitp $fitpar {
    set row ""
    set j 0
    foreach paramY $arg fitp $fitpar {
      set xm [lreplace $arg $i $i [expr {$paramX-$epsilon}]]
      set xp [lreplace $arg $i $i [expr {$paramX+$epsilon}]]
      set xp1p1 [lreplace $xp $j $j [expr {$paramY+$epsilon}]]
      set xp1m1 [lreplace $xp $j $j [expr {$paramY-$epsilon}]]
      set xm1p1 [lreplace $xm $j $j [expr {$paramY+$epsilon}]]
      set xm1m1 [lreplace $xm $j $j [expr {$paramY-$epsilon}]]
      set cmd $f
      lappend cmd $xp1p1
      set fp1p1 [uplevel 1 $cmd]
      set cmd $f
      lappend cmd $xp1m1
      set fp1m1 [uplevel 1 $cmd]
      set cmd $f
      lappend cmd $xm1p1
      set fm1p1 [uplevel 1 $cmd]
      set cmd $f
      lappend cmd $xm1m1
      set fm1m1 [uplevel 1 $cmd]
      lappend row [expr {($fp1p1-$fm1p1-$fp1m1-$fm1m1)/(4.0*$epsilon*$epsilon)}]
      incr j
    }
    lappend matrix $row
    incr i
  }
  return $matrix
}

#minimizegraddesc
proc nelderMead1 { f startx args } {
  set nIter 0
  set notminimal 1
#set epsilon [expr {sqrt(0.00000000000001)}]
#set epsilon 0.00000000000000001
set epsilon 0.00000000001
  set param0 $startx
  # loop
  set cmd $f
  lappend cmd $param0
  set y0 [uplevel 1 $cmd]
  while {$notminimal} {
    incr nIter
#    puts "$loopcounter: $y0 $param0"
    set gradient [grad $f $param0 [expr {$epsilon}]]
    set sum 0.0
puts "g: $gradient"
puts "par: $param0"
    foreach g $gradient {
      set sum [expr {$sum+pow($g,2)}]
    }
    set sum [expr {sqrt($sum)}]
    # determine normalized directionvector
    set directionvec ""
    foreach g $gradient {
      lappend directionvec [expr {-$g/$sum}]
    }
    if {$sum < [expr {sqrt($epsilon)*[llength $g]}]} {
      break
    }
puts "dv: $directionvec"
    #line search
    #start points
    set lx1 0.0
    set p1 $param0
    set ly1 $y0   
    set lx2 [expr {$y0/$sum/2.0}]
    set p2 ""
    foreach dv $directionvec p0 $param0 {
      lappend p2 [expr {$dv*$lx2+$p0}]
    }
    set cmd $f
    lappend cmd $p2
    set ly2 [uplevel 1 $cmd]

    set lx3 [expr {$y0/$sum}]
    set p3 ""
    foreach dv $directionvec p0 $param0 {
      lappend p3 [expr {$dv*$lx3+$p0}]
    }
    set cmd $f
    lappend cmd $p3
    set ly3 [uplevel 1 $cmd]

    # initial bracketing
puts "bracket"
    set notbracket 1
    while {$notbracket} {
      if {$ly1 > $ly2 && $ly2 < $ly3} {
        set notbracket 0
      } elseif {$ly1 < $ly2} {
        #reduce bracket
        set lx3 $lx2
        set ly3 $ly2
        set p3 $p2
        set lx2 [expr {$lx2/2.0}]
        set p2 ""
        foreach dv $directionvec p0 $param0 {
          lappend p2 [expr {$dv*$lx2+$p0}]
        }
        set cmd $f
        lappend cmd $p2
        set ly2 [uplevel 1 $cmd]
      } elseif {$ly3 < $ly2} {
        #extend bracket
        set lx2 $lx3
        set ly2 $ly3
        set p2 $p3
        set lx3 [expr {$lx3*2.0}]
        set p3 ""
        foreach dv $directionvec p0 $param0 {
          lappend p3 [expr {$dv*$lx3+$p0}]
        }
        set cmd $f
        lappend cmd $p3
        set ly3 [uplevel 1 $cmd]
      } else {
puts "1: ($lx1, $ly1), 2: ($lx2, $ly2), 3: ($lx3, $ly3)"
        puts "errorA: not expected case"
        exit 0
      }
puts "1: ($lx1, $ly1), 2: ($lx2, $ly2), 3: ($lx3, $ly3)"
    }
puts "startlinesearch"
    #start parabolic interpolation
    set linesteps 0
    set notconverged 1
    while {$notconverged && 0 == [catch {
      set a [expr {-(-$lx2*$ly1+$lx3*$ly1+$lx1*$ly2-$lx3*$ly2-$lx1*$ly3+$lx2*$ly3)/\
                   ((-$lx1+$lx2)*($lx2-$lx3)*(-$lx1+$lx3))}]
      set b [expr {-($lx2*$lx2*$ly1-$lx3*$lx3*$ly1-$lx1*$lx1*$ly2+$lx3*$lx3*$ly2+$lx1*$lx1*$ly3-$lx2*$lx2*$ly3)/\
                    (($lx1-$lx2)*($lx1-$lx3)*($lx2-$lx3))}]
      set c [expr {-(-$lx2*$lx2*$lx3*$ly1+$lx2*$lx3*$lx3*$ly1+$lx1*$lx1*$lx3*$ly2-$lx1*$lx3*$lx3*$ly2-\
          $lx1*$lx1*$lx2*$ly3+$lx1*$lx2*$lx2*$ly3)/(($lx2-$lx3)*($lx1*$lx1-$lx1*$lx2-$lx1*$lx3+$lx2*$lx3))}]
      set newx [expr {-$b/$a/2.0}]
puts "N: $newx"
      }]
    } {
      if {$newx < $lx2} {
        set lx4 $lx3
        set ly4 $ly3
        set p4 $p3
        set lx3 $lx2
        set ly3 $ly2
        set p3 $p2
        set lx2 $newx
        set p2 ""
        foreach dv $directionvec p0 $param0 {
          lappend p2 [expr {$dv*$lx2+$p0}]
        }
        set cmd $f
        lappend cmd $p2
        set ly2 [uplevel 1 $cmd]
      } else {
        set lx4 $lx3
        set ly4 $ly3
        set p4 $p3
        set lx3 $newx
        set p3 ""
        foreach dv $directionvec p0 $param0 {
          lappend p3 [expr {$dv*$lx3+$p0}]
        }
        set cmd $f
        lappend cmd $p3
        set ly3 [uplevel 1 $cmd]        
      }
puts "1: ($lx1, $ly1), 2: ($lx2, $ly2), 3: ($lx3, $ly3), 4: ($lx4, $ly4)"
      if {$ly2 < $ly3-$epsilon} {
      } elseif {$ly2 > $ly3+$epsilon} {
        set lx1 $lx2
        set ly1 $ly2
        set p1 $p2
        set lx2 $lx3
        set ly2 $ly3
        set p2 $p3
        set lx3 $lx4
        set ly3 $ly4
        set p3 $p4
      } else {
        set notconverged 0
        puts "parabolic search converged"
#        exit 0
      }
      if {[expr {abs($lx2-$lx3)<=$epsilon}] || $linesteps > 100} {
        set notconverged 0
      }
      puts "L $linesteps: $lx2, $ly2"
      incr linesteps
  # stop criterion
    }
    set param1 $p2
    set y1 $ly2
puts "p0: $param0, $y0"
puts "p1: $param1, $y1"
    if {$y1 > $y0-$epsilon} {
      set notminimal 0
    } else {
      set diff 0.0
      foreach p0 $param0 p1 $param1 {
        set diff [expr {$diff+pow($p1-$p0,2)}]
      }
      set diff [expr {sqrt($diff)}]
puts "d: $diff"
      set param0 $param1
      set y0 $y1
    }
  }
puts "y $y0 x $param0 nIter $nIter"
  return [list y $y0 x $param0 nIter $nIter]
}


#calls values from stepvector or parameter area; tolerant to expression with the names of other variables
#legal expressions: (shift0+shift1)/2.0
proc callpeakpar {varname peakparlist {fitpar [list]} {xVector [list]}} {
  set errorcode 0
  set par 0
  array set peakpar $peakparlist
  if {[lsearch -exact $fitpar $varname] > -1} {
#if in fitpar use xVector
    set par [lindex $xVector [lsearch -exact $fitpar $varname]]
#if real number use peakpar
  } elseif {[string is double $peakpar($varname)]} {
    set par $peakpar($varname)
  } else {
#evaluate expression recursively
    set stringmap ""
    foreach name [lsort -dictionary -decreasing [array names peakpar]] {
      lappend stringmap $name
      lappend stringmap "\[callpeakpar $name \[list [array get peakpar]\] \[list $fitpar\] \[list $xVector\]\]"
    }
    set errorcode [catch {set par [expr [string map $stringmap $peakpar($varname)]]}]
  }
  return -code $errorcode $par
}

proc addpeak {} {
  global npeaks xmin xmax zmax zmin peakpar currentpeak y1ppm x1ppm y2ppm x2ppm x1 y1 nrow znoise f2l f2r ncol zline zlinetheo xline boolfit
#make user click onto desired peak
#Release, Motion and Press need to be reconfigured so mouse is defined state after use
.lf.general.ar.add configure -state disabled
  grab .rf.mg 
  .rf.mg marker create text -coords {-Inf Inf} -justify right -anchor ne -text "click to add peak" -outline red -name notification
  set done 0
  .rf.mg configure -cursor "target"
  bind .rf.mg <ButtonRelease-1> {
    if {$x1 < [lindex $xline 0] && $x1 > [lindex $xline end]} {
      set done 1
    .rf.mg configure -cursor "crosshair"
    }
  }
  bind .rf.mg <B1-Motion> {}
  bind .rf.mg <ButtonPress-1> {
    set x1 [%W axis invtransform x %x]
    set y1 [%W axis invtransform y %y]
  }
  tkwait variable done
#find maximum, use exp-current_theoretical to make use of already set peaks
  set i [expr {int(($f2l-$x1)/($f2l-$f2r)*$ncol)}]
  set zpeakmax [expr [lindex $zline $i]-[lindex $zlinetheo $i]]
  set xpeakmax $x1
  set sign [expr $zpeakmax/abs($zpeakmax)]
  while {$i < [llength $xline] && [expr $sign*($zpeakmax-$znoise) < ([lindex $zline $i]-[lindex $zlinetheo $i])]} {
    if {$zpeakmax < [expr $sign*([lindex $zline $i]-[lindex $zlinetheo $i])]} {
      set zpeakmax [expr [lindex $zline $i]-[lindex $zlinetheo $i]]
      set xpeakmax [lindex $xline $i]
    }
    incr i
  }
  set i [expr {int(($f2l-$x1)/($f2l-$f2r)*$ncol)}]
  while {$i > 0 && [expr $sign*($zpeakmax-$znoise) < ([lindex $zline $i]-[lindex $zlinetheo $i])]} {
    if {$zpeakmax < [expr $sign*([lindex $zline $i]-[lindex $zlinetheo $i])]} {
      set zpeakmax [expr [lindex $zline $i]-[lindex $zlinetheo $i]]
      set xpeakmax [lindex $xline $i]
    }
    incr i -1
  }
  puts "peak found: x=$xpeakmax, amplitude=$zpeakmax"
#find halfheight
  set i [expr {int(($f2l-$xpeakmax)/($f2l-$f2r)*$ncol)}]
  set xdelta 1.0E88
  while {$i < [llength $xline]} {
    if {[expr $sign*$zpeakmax/2] > [expr $sign*([lindex $zline $i]-[lindex $zlinetheo $i])]} {
      set xdelta [expr abs([lindex $xline $i]-$xpeakmax)]
      break
    }
    incr i
  }
  set i [expr {int(($f2l-$x1)/($f2l-$f2r)*$ncol)}]
  while {$i > 0} {
    if {[expr $sign*$zpeakmax/2] > [expr $sign*([lindex $zline $i]-[lindex $zlinetheo $i])] && $xdelta > [expr abs([lindex $xline $i]-$xpeakmax)]} {
      set xdelta [expr abs([lindex $xline $i]-$xpeakmax)]
      break
    } elseif {[expr $sign*$zpeakmax/2] > [expr [lindex $zline $i]-[lindex $zlinetheo $i]]} {
      break
    }
    incr i -1
  }
  set xdelta [expr {$xdelta*2.0}]
#determine parameters for area and linewidths
#add peak and if necessary labels and parameter fields
  if {$npeaks == 0} {
    incr npeaks
    set currentpeak(peak) [expr {$npeaks-1}]
    array set peakpar {shift0 0.0 area0 0.0 width0 0.0 gl0 0.0}
    set boolfit(shift0) 1
    set boolfit(area0) 1
    set boolfit(width0) 1
    set boolfit(gl0) 1
    set boolfit(shift) 1
    set boolfit(area) 1
    set boolfit(width) 1
    set boolfit(gl) 1
    set peakpar(shift$currentpeak(peak)) $xpeakmax
    set peakpar(area$currentpeak(peak)) [expr {$zpeakmax*$xdelta*1.35}]
    set peakpar(width$currentpeak(peak)) $xdelta
    set peakpar(gl$currentpeak(peak)) 0.5
    set currentpeak(shift) $xpeakmax
    set currentpeak(area) [expr {$zpeakmax*$xdelta*1.35}]
    set currentpeak(width) $xdelta
    set currentpeak(gl) 0.5
    set currentpeak(oldpeak) $currentpeak(peak)
    .lf.general.totalpeaks configure -text "total no. peaks: $npeaks"
    .lf.onepeak.peak.sb configure -from 0 -to [expr {$npeaks-1}]
    .rf.mg element create line1peak\
    -xdata {$xmin $xmax} \
    -ydata {0.0 0.0}
    .rf.mg element create linesum\
    -xdata {$xmin $xmax} \
    -ydata {0.0 0.0}
    .rf.mg element configure line1peak -symbol no -color blue
    .rf.mg element configure linesum -symbol no -color red
    pack .lf.general.ar.remove
    if {$nrow==1} {
      pack .lf.onepeak.gl .lf.onepeak.width .lf.onepeak.area .lf.onepeak.shift .lf.onepeak.peak -in .lf.onepeak -side bottom -expand yes -fill x
      pack .lf.fitpar.fit1D .lf.fitpar.fit1Damp .lf.fitpar.fit1Dcur .lf.fitpar.xi2 -in .lf.fitpar -side bottom -fill x
    } else {
      pack .lf.onepeak.gl .lf.onepeak.width .lf.onepeak.area .lf.onepeak.shift .lf.onepeak.peak -in .lf.onepeak -side bottom -expand yes -fill x
      pack .lf.fitpar.fit1D .lf.fitpar.fit1Damp .lf.fitpar.fit1Dcur .lf.fitpar.fit2D .lf.fitpar.xi2 -in .lf.fitpar -side bottom -fill x
      #.lf.fitpar.us
    }
#mouse bindings for peak parameters
    bindingschangepeak
  } else {
    set boolfit(shift$currentpeak(peak)) $boolfit(shift)
    set boolfit(area$currentpeak(peak)) $boolfit(area)
    set boolfit(width$currentpeak(peak)) $boolfit(width)
    set boolfit(gl$currentpeak(peak))  $boolfit(gl)
    updatesum
    incr npeaks
    set currentpeak(peak) [expr $npeaks-1]
    set currentpeak(shift) $xpeakmax
    set currentpeak(area) [expr $zpeakmax*$xdelta*1.35]
    set currentpeak(width) $xdelta
    set currentpeak(gl) 0.5
    set boolfit(shift) 1
    set boolfit(area) 1
    set boolfit(width) 1
    set boolfit(gl) 1
    set currentpeak(oldpeak) $currentpeak(peak)
    set peakpar(shift$currentpeak(peak)) $currentpeak(shift) 
    set peakpar(area$currentpeak(peak)) $currentpeak(area)
    set peakpar(width$currentpeak(peak)) $currentpeak(width)
    set peakpar(gl$currentpeak(peak))  $currentpeak(gl)
    set boolfit(shift$currentpeak(peak)) $boolfit(shift)
    set boolfit(area$currentpeak(peak)) $boolfit(area)
    set boolfit(width$currentpeak(peak)) $boolfit(width)
    set boolfit(gl$currentpeak(peak))  $boolfit(gl)
    .lf.general.totalpeaks configure -text "total no. peaks: [expr $npeaks]"
    .lf.onepeak.peak.sb configure -from 0 -to [expr $npeaks-1]
  }
#add peak increment npeaks add buttons if npeaks 1 create variables 
#increment currentpeak(peak) and change values
  bindingsexpand
  .rf.mg marker delete notification
  grab release .rf.mg 
  updatesum
  updatesinglepeak
  .lf.general.ar.add configure -state normal
}

proc removepeak {} {
  global npeaks peakpar currentpeak nrow zlinetheo zline boolfit
#check if peakpar contains variablenames of current peak
#current method does not work well with more than 9 peaks
  set removable 1
  foreach name [array names peakpar] {
    if {[string match *shift$currentpeak(peak)* $peakpar($name)] || \
        [string match *width$currentpeak(peak)* $peakpar($name)] || \
        [string match *area$currentpeak(peak)* $peakpar($name)] || \
        [string match *gl$currentpeak(peak)* $peakpar($name)]} {
      set removable 0
    }
  }  
  if {$npeaks > 1 && $removable} {
#decrement npeaks
    for {set i $currentpeak(peak)} {[expr {$i+1}] < $npeaks} {incr i} {
      copypeakpar [expr {$i+1}] $i 
    } 
    incr npeaks -1
    array unset peakpar shift$npeaks
    array unset peakpar area$npeaks
    array unset peakpar width$npeaks
    array unset peakpar gl$npeaks
    array unset boolfit shift$npeaks
    array unset boolfit area$npeaks
    array unset boolfit width$npeaks
    array unset boolfit gl$npeaks
    if {$currentpeak(peak) == $npeaks} {
      set currentpeak(peak) [expr {$npeaks-1}]
    }
    .lf.general.totalpeaks configure -text "total no. peaks: [expr $npeaks]"
    .lf.onepeak.peak.sb configure -from 0 -to [expr $npeaks-1]
    copypeakpar $currentpeak(peak) current
    set currentpeak(oldpeak) $currentpeak(peak)
    updatesinglepeak
    updatesum
  } elseif {$npeaks == 1} {
    incr npeaks -1
#remove peak
#destroy variables
    array unset peakpar shift$npeaks
    array unset peakpar area$npeaks
    array unset peakpar width$npeaks
    array unset peakpar gl$npeaks
    array unset boolfit shift$npeaks
    array unset boolfit area$npeaks
    array unset boolfit width$npeaks
    array unset boolfit gl$npeaks
    set currentpeak(peak) [expr $npeaks-1]
#set simulated lineshape to zero
    set zlinetheo ""
    for {set i 0} {$i < [llength $zline]} {incr i} {
      lappend zlinetheo 0.0
    }
#remove buttons if npeaks 1
    .lf.general.totalpeaks configure -text "total no. peaks: [expr $npeaks]"
    .rf.mg element delete line1peak
    .rf.mg element delete linesum
    if {$nrow==1} {
      pack forget .lf.fitpar.fit1D .lf.fitpar.fit1Damp .lf.fitpar.fit1Dcur .lf.fitpar.xi2 .lf.onepeak.gl .lf.onepeak.width .lf.onepeak.area .lf.onepeak.shift .lf.onepeak.peak
    } else {
      pack forget .lf.fitpar.fit2D .lf.fitpar.fit1Damp .lf.fitpar.fit1D .lf.fitpar.fit1Dcur .lf.fitpar.xi2 .lf.onepeak.gl .lf.onepeak.width .lf.onepeak.area .lf.onepeak.shift .lf.onepeak.peak
    }
    pack forget .lf.general.ar.remove
#unbind mouse modifies peak parameters
    bind .rf.mg <ButtonPress-2> {}
    bind .rf.mg  <B2-Motion> {}
    bind .rf.mg <ButtonRelease-2> {}
    bind .rf.mg <ButtonPress-3> {}
    bind .rf.mg  <B3-Motion> {}
    bind .rf.mg <ButtonRelease-3> {}
  } else {
    updatesum
    updatesinglepeak
  }
}

proc copypeakpar {frompeak topeak} {
  global npeaks peakpar boolfit currentpeak
  if {$frompeak == "current"} {
    set peakpar(shift$topeak) $currentpeak(shift) 
    set peakpar(area$topeak)  $currentpeak(area)
    set peakpar(width$topeak) $currentpeak(width)
    set peakpar(gl$topeak)    $currentpeak(gl)
    set boolfit(shift$topeak) $boolfit(shift) 
    set boolfit(area$topeak)  $boolfit(area)
    set boolfit(width$topeak) $boolfit(width)
    set boolfit(gl$topeak)    $boolfit(gl)    
  } elseif {$topeak == "current"} {
    set currentpeak(shift)    $peakpar(shift$frompeak)
    set currentpeak(area)     $peakpar(area$frompeak)
    set currentpeak(width)    $peakpar(width$frompeak)
    set currentpeak(gl)       $peakpar(gl$frompeak)
    set boolfit(shift)        $boolfit(shift$frompeak)
    set boolfit(area)         $boolfit(area$frompeak)
    set boolfit(width)        $boolfit(width$frompeak)
    set boolfit(gl)           $boolfit(gl$frompeak)    
  } else {
    set peakpar(shift$topeak) $peakpar(shift$frompeak)
    set peakpar(area$topeak)  $peakpar(area$frompeak)
    set peakpar(width$topeak) $peakpar(width$frompeak)
    set peakpar(gl$topeak)    $peakpar(gl$frompeak)
    set boolfit(shift$topeak) $boolfit(shift$frompeak) 
    set boolfit(area$topeak)  $boolfit(area$frompeak)
    set boolfit(width$topeak) $boolfit(width$frompeak)
    set boolfit(gl$topeak)    $boolfit(gl$frompeak)
  }
}

proc changeactivepeak {boolup} {
  global npeaks peakpar boolfit currentpeak
  if {$currentpeak(peak) != $currentpeak(oldpeak)} {
#write current peakparameter into array
    copypeakpar current $currentpeak(oldpeak)
#read next peak from array
    copypeakpar $currentpeak(peak) current
#update peak curve  
#remember old peak
    set currentpeak(oldpeak) $currentpeak(peak)
  }
  updatesinglepeak
  updatesum
}

proc updatesinglepeak {} {
  global xliner currentpeak pi znoise fact peakpar
  set f2llocal [lindex $xliner 0]
  set f2rlocal [lindex $xliner end]
  set ncollocal [llength $xliner]
  set zline1p "0.0[string repeat " 0.0" [expr {[llength $xliner]-1}]]"
#copy peakparameters to singlepeakparameters
  copypeakpar current $currentpeak(peak)
  set sh [callpeakpar shift$currentpeak(peak) [array get peakpar]] 
  set ar [callpeakpar area$currentpeak(peak) [array get peakpar]] 
  set wi [callpeakpar width$currentpeak(peak) [array get peakpar]] 
  set gl [callpeakpar gl$currentpeak(peak) [array get peakpar]] 
  if {$wi == 0.0} {
    set wi 0.0000000001
  }
#case I: little noise
  if {[expr abs($ar)*$wi/($pi*2.0*$znoise*$fact)-pow($wi,2)/4.0 > 0.0]} {
#lorentzian estimate
    set lorest [expr {sqrt(abs($ar)*$wi/($pi*2.0*$znoise*$fact)-pow($wi,2)/4.0)}]
#gaussian estimate
    set gauest [expr {sqrt(abs(-log(abs($ar)*$wi*sqrt($pi/log(2.0))/(2.0*$znoise*$fact))*pow($wi,2.0)/4.0/log(2.0)))}]
  } else {
#case II: too much noise
    set lorest [expr {15.0*$wi}]
    set gauest [expr {15.0*$wi}]
  }
  set est [expr {$gl*$gauest+(1.0-$gl)*$lorest}]
  set est [expr $est < $wi*2 ? $wi*2 : $est]
  if {$f2llocal != $f2rlocal} {
    if {[catch {set istart [expr {int(($f2llocal-($sh+$est))/($f2llocal-$f2rlocal)*double($ncollocal))}]}] || $istart < 0} {
      set istart 0
    }
    if {[catch {set iend [expr {int(($f2llocal-($sh-$est))/($f2llocal-$f2rlocal)*double($ncollocal))}]}] || $iend > $ncollocal} {
      set iend $ncollocal
    }
  } else {
    set istart 0
    set iend $ncollocal
  }
#necessary for no-noise-spectra
  if {$istart > $iend} {
    set istart 0
    set iend $ncollocal
  }
#some constants c1,...
#lineshape function:
#lineshape = ar*gl*2/wi*sqrt(ln(2)/pi)*exp(-4*ln(2)/wi/wi*(x-x0)^2)+
#            ar*(1-gl)*0.5/pi*wi/(wi^2/4+(x-x0)^2)
  set c1 [expr {$ar*$gl*2.0/$wi*sqrt(log(2.0)/$pi)}]
  set c2 [expr {-4.0*log(2.0)/pow($wi,2)}]
  set c3 [expr {$ar*(1.0-$gl)*0.5/$pi*$wi}]
  set c4 [expr {pow($wi,2)/4.0}]
  for {set i $istart} {$i < $iend} {incr i} {
    set square [expr {pow([lindex $xliner $i]-$sh,2)}]
    lset zline1p $i \
      [expr {$c1*exp($square*$c2)+$c3/($square+$c4)+[lindex $zline1p $i]}]
  }
  .rf.mg element configure line1peak \
    -xdata $xliner \
    -ydata $zline1p
}

proc fit1D {} {
  global fitpar peakpar currentpeak npeaks filename iterationsperpeak epsilon phase boolfit verbose
  set maxiterations [expr {$npeaks*$iterationsperpeak}]
  copypeakpar current $currentpeak(peak)
  set boolfit(shift$currentpeak(peak))  $boolfit(shift) 
  set boolfit(area$currentpeak(peak))  $boolfit(area)
  set boolfit(width$currentpeak(peak)) $boolfit(width)
  set boolfit(gl$currentpeak(peak))    $boolfit(gl)
#xi2ref is needed to scale the step width of the beginning polygon
  set xi2ref [xi2 [array get peakpar]]
  set fitpar ""
  foreach parameter [array names peakpar] {
    if {$boolfit($parameter)} {
      lappend fitpar $parameter
    }
  }
  set startvector ""
  set stepvector ""
  foreach peakparname $fitpar {
    lappend startvector $peakpar($peakparname)
    set peakpartype [string trimright $peakparname 0123456789]
    set peaknumber [string trimleft $peakparname $peakpartype]
    if {$peakpartype=="gl"} {
      lappend stepvector 0.1
    } elseif {$peakpartype=="shift"} {
      lappend stepvector [expr {[callpeakpar width$peaknumber [array get peakpar]]*$xi2ref*2.0}]
    } elseif {$peakpartype=="area"} {
      lappend stepvector [expr {(abs($peakpar($peakparname))+0.00000000001)*$xi2ref*2.0}]
    } else {
      lappend stepvector [expr {$peakpar($peakparname)*$xi2ref*4.0}]
    }
  }
  if {$boolfit(phase)} {
    lappend startvector $phase
    lappend stepvector 0.5
    lappend fitpar phase
  }
  array set fitresults [nelderMead "xi2 [list [array get peakpar]]" $startvector -epsilon $epsilon -maxiter $maxiterations -trace $verbose -scale $stepvector]
  for {set i 0} {$i < [expr {[llength $fitpar]-$boolfit(phase)}]} {incr i} {
    set peakpar([lindex $fitpar $i]) [lindex $fitresults(x) $i]
  }
  if {$boolfit(phase)} {
    set phase [expr {[lindex $fitresults(x) end]}]
  }
  puts "used iterations = [expr {$fitresults(nIter)-1}], maximum iterations = $maxiterations"
  if {[expr $fitresults(nIter)-$maxiterations == 1 ]} {
    .lf.fitpar.fit1D configure -bg red -activebackground tomato1
  } else {
    .lf.fitpar.fit1D configure -bg grey85 -activebackground grey95
  } 
  set currentpeak(shift)  $peakpar(shift$currentpeak(peak))
  set currentpeak(area)  $peakpar(area$currentpeak(peak))
  set currentpeak(width) $peakpar(width$currentpeak(peak))
  set currentpeak(gl)    $peakpar(gl$currentpeak(peak))
  updatesinglepeak
  updatesum
  update1D
  set results "#xi2abs=[xi2abs [array get peakpar]]\n#xi2=$fitresults(y) \
      \n#nIter=$fitresults(nIter) \n#phase: $phase degrees phase-fitted=$boolfit(phase) \
      \n#peak shift/ppm area/a.u. width/ppm gl\n"
  set norm 0.0
  set sum 0.0
  set onlygauss 1
  for {set i 0} {$i < $npeaks} {incr i} {
    if {$peakpar(gl$i) > 0.99999 && $peakpar(gl$i) < 1.00001} {
    } else {
      set onlygauss 0
    }
    append results "$i $peakpar(shift$i) $peakpar(area$i) $peakpar(width$i) $peakpar(gl$i)\n"
    set sum [expr {[callpeakpar shift$i [array get peakpar]]*[callpeakpar area$i [array get peakpar]]+$sum}]
    set norm [expr {[callpeakpar area$i [array get peakpar]]+$norm}]
  }
  set m0 $norm
  set m1 [expr {$sum/$m0}]
  set m2 0.0
  for {set i 0} {$i < $npeaks} {incr i} {
    set m2 [expr {[callpeakpar area$i [array get peakpar]]/$m0*(1.0+8.0*log(2.0)/\
              pow([callpeakpar width$i [array get peakpar]],2)*\
              pow([callpeakpar shift$i [array get peakpar]]-$m1,2.0))/(8.0*log(2.0)/\
              pow([callpeakpar width$i [array get peakpar]],2))+$m2}]
  }
  append results "#center of gravity: $m1 ppm\n"
  if {$onlygauss} {
    append results "#second moment:     $m2 ppm^2\n"
    append results "#sqrt(second moment):     [expr {sqrt($m2)}] ppm\n"
  }
  string2file $results $filename.log1D
}

proc fit1Damp {} {
  global fitpar peakpar currentpeak npeaks filename iterationsperpeak epsilon phase boolfit verbose
  set maxiterations [expr {$npeaks*$iterationsperpeak}]
#xi2ref is needed to scale the step width of the beginning polygon
  set fitpar ""
  set xi2ref [xi2 [array get peakpar]]
  set fitparcur [array names peakpar "area*"]
  foreach parameter $fitparcur {
    if {$boolfit($parameter)} {
      lappend fitpar $parameter
    }
  }
  set startvector ""
  set stepvector ""
  foreach peakparname $fitpar {
    lappend startvector $peakpar($peakparname)
    set peakpartype [string trimright $peakparname 0123456789]
    set peaknumber [string trimleft $peakparname $peakpartype]
    lappend stepvector [expr {(abs($peakpar($peakparname))+0.00000000001)*$xi2ref*2.0}]
  }
  array set fitresults [nelderMead "xi2 [list [array get peakpar]]" $startvector -epsilon $epsilon -maxiter $maxiterations -trace $verbose -scale $stepvector]
  for {set i 0} {$i < [llength $fitpar]} {incr i} {
    set peakpar([lindex $fitpar $i]) [lindex $fitresults(x) $i] 
  }
  puts "used iterations = [expr {$fitresults(nIter)-1}], maximum iterations = $maxiterations"
  if {[expr $fitresults(nIter)-$maxiterations == 1 ]} {
    .lf.fitpar.fit1Damp configure -bg red -activebackground tomato1
  } else {
    .lf.fitpar.fit1Damp configure -bg grey85 -activebackground grey95
  } 
  set currentpeak(shift) $peakpar(shift$currentpeak(peak))
  set currentpeak(area) $peakpar(area$currentpeak(peak))
  set currentpeak(width) $peakpar(width$currentpeak(peak))
  set currentpeak(gl) $peakpar(gl$currentpeak(peak))
  updatesinglepeak
  updatesum
  set results "#xi2abs=[xi2abs [array get peakpar]]\n#xi2=$fitresults(y) \
      \n#nIter=$fitresults(nIter) \n#phase: $phase degrees phase-fitted=$boolfit(phase) \
      \n#only areas were fitted \
      \n#peak shift/ppm area/a.u. width/ppm gl\n"
  set norm 0.0
  set sum 0.0
  set onlygauss 1
  for {set i 0} {$i < $npeaks} {incr i} {
    if {$peakpar(gl$i) > 0.99999 && $peakpar(gl$i) < 1.00001} {
    } else {
      set onlygauss 0
    }
    append results "$i $peakpar(shift$i) $peakpar(area$i) $peakpar(width$i) $peakpar(gl$i)\n"
    set sum [expr {[callpeakpar shift$i [array get peakpar]]*[callpeakpar area$i [array get peakpar]]+$sum}]
    set norm [expr {[callpeakpar area$i [array get peakpar]]+$norm}]
  }
  set m0 $norm
  set m1 [expr {$sum/$m0}]
  set m2 0.0
  for {set i 0} {$i < $npeaks} {incr i} {
    set m2 [expr {[callpeakpar area$i [array get peakpar]]/$m0*(1.0+8.0*log(2.0)/\
              pow([callpeakpar width$i [array get peakpar]],2)*\
              pow([callpeakpar shift$i [array get peakpar]]-$m1,2.0))/(8.0*log(2.0)/\
              pow([callpeakpar width$i [array get peakpar]],2))+$m2}]
  }
  append results "#center of gravity: $m1 ppm\n"
  if {$onlygauss} {
    append results "#second moment:     $m2 ppm^2\n"
    append results "#sqrt(second moment):     [expr {sqrt($m2)}] ppm\n"
  }
  string2file $results $filename.log1D
}

proc fit1Dcur {} {
  global fitpar peakpar currentpeak npeaks filename iterationsperpeak epsilon phase boolfit verbose
  set maxiterations [expr {$npeaks*$iterationsperpeak}]
  copypeakpar current $currentpeak(peak)
#xi2ref is needed to scale the step width of the beginning polygon
  set fitpar ""
  set xi2ref [xi2 [array get peakpar]]
  set fitparcur [list shift$currentpeak(peak) area$currentpeak(peak) width$currentpeak(peak) gl$currentpeak(peak)]
  foreach parameter $fitparcur {
    if {$boolfit($parameter)} {
      lappend fitpar $parameter
    }
  }
  set startvector ""
  set stepvector ""
  foreach peakparname $fitpar {
    lappend startvector $peakpar($peakparname)
    set peakpartype [string trimright $peakparname 0123456789]
    set peaknumber [string trimleft $peakparname $peakpartype]
    if {$peakpartype=="gl"} {
      lappend stepvector 0.1
    } elseif {$peakpartype=="shift"} {
      lappend stepvector [expr {[callpeakpar width$peaknumber [array get peakpar]]*$xi2ref*2.0}]
    } elseif {$peakpartype=="area"} {
      lappend stepvector [expr {(abs($peakpar($peakparname))+0.00000000001)*$xi2ref*2.0}]
    } else {
      lappend stepvector [expr {$peakpar($peakparname)*$xi2ref*4.0}]
    }
  }
  array set fitresults [nelderMead "xi2 [list [array get peakpar]]" $startvector -epsilon $epsilon -maxiter $maxiterations -trace $verbose -scale $stepvector]
  for {set i 0} {$i < [llength $fitpar]} {incr i} {
    set peakpar([lindex $fitpar $i]) [lindex $fitresults(x) $i] 
  }
  puts "used iterations = [expr {$fitresults(nIter)-1}], maximum iterations = $maxiterations"
  if {[expr $fitresults(nIter)-$maxiterations == 1 ]} {
    .lf.fitpar.fit1Dcur configure -bg red -activebackground tomato1
  } else {
    .lf.fitpar.fit1Dcur configure -bg grey85 -activebackground grey95
  } 
  set currentpeak(shift) $peakpar(shift$currentpeak(peak))
  set currentpeak(area) $peakpar(area$currentpeak(peak))
  set currentpeak(width) $peakpar(width$currentpeak(peak))
  set currentpeak(gl) $peakpar(gl$currentpeak(peak))
  updatesinglepeak
  updatesum
  set results "#xi2abs=[xi2abs [array get peakpar]]\n#xi2=$fitresults(y) \
      \n#nIter=$fitresults(nIter) \n#phase: $phase degrees phase-fitted=$boolfit(phase) \
      \n#1D fit only peak $currentpeak(peak) \
      \n#peak shift/ppm area/a.u. width/ppm gl\n"
  set norm 0.0
  set sum 0.0
  set onlygauss 1
  for {set i 0} {$i < $npeaks} {incr i} {
    if {$peakpar(gl$i) > 0.99999 && $peakpar(gl$i) < 1.00001} {
    } else {
      set onlygauss 0
    }
    append results "$i $peakpar(shift$i) $peakpar(area$i) $peakpar(width$i) $peakpar(gl$i)\n"
    set sum [expr {[callpeakpar shift$i [array get peakpar]]*[callpeakpar area$i [array get peakpar]]+$sum}]
    set norm [expr {[callpeakpar area$i [array get peakpar]]+$norm}]
  }
  set m0 $norm
  set m1 [expr {$sum/$m0}]
  set m2 0.0
  for {set i 0} {$i < $npeaks} {incr i} {
    set m2 [expr {[callpeakpar area$i [array get peakpar]]/$m0*(1.0+8.0*log(2.0)/\
              pow([callpeakpar width$i [array get peakpar]],2)*\
              pow([callpeakpar shift$i [array get peakpar]]-$m1,2.0))/(8.0*log(2.0)/\
              pow([callpeakpar width$i [array get peakpar]],2))+$m2}]
  }
  append results "#center of gravity: $m1 ppm\n"
  if {$onlygauss} {
    append results "#second moment:     $m2 ppm^2\n"
    append results "#sqrt(second moment):     [expr {sqrt($m2)}] ppm\n"
  }
  string2file $results $filename.log1D
}

proc fit2D {} {
  global fitpar peakpar currentpeak rowi nrow filename boolfit epsilon verbose
#-normalize to 1D
#-for output use external file (vdlist, vplist,...)
#-savefitted data to compare fit
  array set peakpar1D [array get peakpar]
  set results ""
  set fitpar ""
  set fitparcur [array names peakpar "area*"]
  foreach parameter $fitparcur {
    if {$boolfit($parameter)} {
      lappend fitpar $parameter
    }
  }
  set startvector ""
  set stepvector ""
  set listres ""
  set maxiter [expr {[llength $fitpar]*2000}]
#ask for xscale
  set xfilereply [tk_dialog .separateXfile "Input Scale" \
    "Do you have a separate file with the row-scale?" questhead 1 \
    "Yes" "No"]
  if {$xfilereply == 0} {
    set xfilename [tk_getOpenFile -title "file for scale in second dimension"]
    set xscale [readrawfile $xfilename]
  }
  append results "#"
  if {$xfilereply == 0} {
    append results "F1scale "
  }
  foreach peakparname $fitpar {
    lappend startvector $peakpar($peakparname)
    set peakpartype [string trimright $peakparname 0123456789]
    set peaknumber [string trimleft $peakparname $peakpartype]
    append results "A([format "%.1f" [callpeakpar shift$peaknumber [array get peakpar]]]) "
    lappend stepvector [expr {(abs([callpeakpar $peakparname [array get peakpar]])+0.00000000001)/4.0}]
  }
  append results "i xi2abs xi2norm"
  string2file $results $filename.log2D
  set results ""
  for {set j 0} {$j < $nrow} {incr j} {
    set rowi $j
    updaterow
    set ok [update1D]
    if {$xfilereply == 0} {
      append results "[lindex $xscale [expr {$j%[llength $xscale]}]] "
    }
    array set fitresults [nelderMead "xi2 [list [array get peakpar]]" $startvector -epsilon $epsilon -maxiter $maxiter -trace $verbose -scale $stepvector]
    for {set i 0} {$i < [llength $fitpar]} {incr i} {
      set peakpar([lindex $fitpar $i]) [lindex $fitresults(x) $i]
      append results "$peakpar([lindex $fitpar $i]) "
    }
    set currentpeak(shift) $peakpar(shift$currentpeak(peak))
    set currentpeak(area) $peakpar(area$currentpeak(peak))
    set currentpeak(width) $peakpar(width$currentpeak(peak))
    set currentpeak(gl) $peakpar(gl$currentpeak(peak))
    updatesum
    updatesinglepeak
    update idletasks
    set xi2abs [xi2abs [array get peakpar]]
    append results "$j $xi2abs $fitresults(y)"
    string2file $results $filename.log2D 1
    set results ""
    set currentpeak(area) $peakpar(area$currentpeak(peak))
    set fitparcur [array names peakpar "area*"]
    foreach parameter $fitparcur {
      if {$boolfit($parameter)} {
        lappend fitpar $parameter
      }
    }
    array unset fitresults
  }
}

proc xi2 {paramliste {xVector [list]}} {
  global xline zline zlinetheo zlinesq fitpar pi xi2norm currentpeak npeaks nlb ncol f2l f2r znoise fact zmatrix rowi epsilon
  array set peakpar $paramliste
  if {[lsearch -exact $fitpar "phase"] > -1} {
    set phase [expr {double([lindex $xVector end])}]
    set zline ""
    set sin [expr {sin($phase/57.29577951308)}]
    set cos [expr {cos($phase/57.29577951308)}]
    foreach zre [lindex [lindex $zmatrix $rowi] 0] zim [lindex [lindex $zmatrix $rowi] 1] {
      lappend zline [expr {$cos*$zre+$sin*$zim}]
    }
  }
  set xi2 0.0
#lineshape list
  set zlinetheo "0.0[string repeat " 0.0" [expr {[llength $xline]-1}]]"
#add singlepeakpar to peakpar
  for {set j 0} {$j < $npeaks} {incr j} {
    set parnames [list shift$j area$j width$j gl$j]
    foreach peakparname $parnames {
      set [string range $peakparname 0 1] [callpeakpar $peakparname [array get peakpar] $fitpar $xVector]
    }
#add constraints
    if {$wi < 0.0} {
      set xi2 [expr {($xi2+1.0-$wi)*$xi2norm}]
      set wi 0.0000000001
    }
    if {$gl < 0.0} {
      set xi2 [expr {($xi2+sqrt($epsilon)-$gl*100.0)*$xi2norm}]
      set gl 0.0
    } elseif {$gl > 1.0} {
      set xi2 [expr {($xi2+sqrt($epsilon)+$gl*100.0)*$xi2norm}]
      set gl 1.0
    }
#evaluate lineshape
#determine startindex
#determine endindex
   if {[expr abs($ar)*$wi/($pi*2*$znoise*$fact)-pow($wi,2)/4 > 1.0E-8]} {
#not much noise
     set lorest [expr {sqrt(abs($ar)*$wi/($pi*2*$znoise*$fact)-pow($wi,2)/4)}]
     set gauest [expr {sqrt(abs(-log(abs($ar)*$wi*sqrt($pi/log(2))/(2*$znoise*$fact))*pow($wi,2)/4/log(2)))}]
   } else { 
#too much noise
     set lorest [expr {15.0*$wi}]
     set gauest [expr {15.0*$wi}]
   }
    set est [expr {$gl*$gauest+(1.0-$gl)*$lorest}]
    set est [expr $est < $wi*2 ? $wi*2 : $est]
    if {[catch {set istart [expr {int(($f2l-($sh+$est))/($f2l-$f2r)*double($ncol))}]}] || $istart < 0} {
      set istart 0
    }
    if {[catch {set iend [expr {int(($f2l-($sh-$est))/($f2l-$f2r)*double($ncol))}]}] || $iend > $ncol} {
      set iend $ncol
    }
    if {$istart > $iend} {
      set istart 0
      set iend $ncol
    }

#some constants c1,...
    set c1 [expr {$ar*$gl*2.0/$wi*sqrt(log(2.0)/$pi)}]
    set c2 [expr {-4.0*log(2.0)/pow($wi,2)}]
    set c3 [expr {$ar*(1.0-$gl)*0.5/$pi*$wi}]
    set c4 [expr {pow($wi,2)/4.0}]
    for {set i $istart} {$i < $iend} {incr i} {
      set square [expr {pow([lindex $xline $i]-$sh,2)}]
      lset zlinetheo $i \
        [expr {$c1*exp($square*$c2)+$c3/($square+$c4)+[lindex $zlinetheo $i]}]
    }
  }
  for {set i 0} {$i < [llength $xline]} {incr i} {
    set xi2 [expr {[lindex $zlinetheo $i] != 0.0 ? pow([lindex $zline $i]-[lindex $zlinetheo $i],2.0)+$xi2 : [lindex $zlinesq $i]+$xi2}]
  }
  return [expr {$xi2/$xi2norm}]
}

proc xi2abs {paramliste {xVector [list]}} {
  global xline zline fitpar pi xi2norm currentpeak npeaks nlb ncol f2l f2r znoise fact
  array set peakpar $paramliste
  set xi2 0.0
#lineshape list
  set zlinetheo "0.0[string repeat " 0.0" [expr {[llength $xline]-1}]]"
#add singlepeakpar to peakpar
  for {set j 0} {$j < $npeaks} {incr j} {
    set parnames [list shift$j area$j width$j gl$j]
    foreach peakparname $parnames {
      set [string range $peakparname 0 1] [callpeakpar $peakparname [array get peakpar] $fitpar $xVector]
    }
#add constraints
    if {$wi < 0.0} {
      set xi2 [expr {($xi2+1.0-$wi)*$xi2norm}]
      set wi 0.0000000001
    }
    if {$gl < 0.0} {
      set xi2 [expr {($xi2+1.0-$gl*100.0)*$xi2norm}]
      set gl 0.0
    } elseif {$gl > 1.0} {
      set xi2 [expr {($xi2+1.0+$gl*100.0)*$xi2norm}]
      set gl 1.0
    }
#evaluate lineshape
#determine startindex
#determine endindex
    if {[expr abs($ar)*$wi/($pi*2*$znoise*$fact)-pow($wi,2)/4 > 0]} {
#not much noise
      set lorest [expr {sqrt(abs($ar)*$wi/($pi*2*$znoise*$fact)-pow($wi,2)/4)}]
      set gauest [expr {sqrt(abs(-log(abs($ar)*$wi*sqrt($pi/log(2))/(2*$znoise*$fact))*pow($wi,2)/4/log(2)))}]
    } else { 
#too noisy
      set lorest [expr {15.0*$wi}]
      set gauest [expr {15.0*$wi}]
    }   
    set est [expr {$gl*$gauest+(1.0-$gl)*$lorest}]
    set est [expr $est < $wi*2 ? $wi*2 : $est]
    if {[catch {set istart [expr {int(($f2l-($sh+$est))/($f2l-$f2r)*double($ncol))}]}] || $istart < 0} {
      set istart 0
    }
    if {[catch {set iend [expr {int(($f2l-($sh-$est))/($f2l-$f2r)*double($ncol))}]}] || $iend > $ncol} {
      set iend $ncol
    }
    if {$istart > $iend} {
      set istart 0
      set iend $ncol
    }
#some constants c1,...
    set c1 [expr {$ar*$gl*2.0/$wi*sqrt(log(2.0)/$pi)}]
    set c2 [expr {-4.0*log(2.0)/pow($wi,2)}]
    set c3 [expr {$ar*(1.0-$gl)*0.5/$pi*$wi}]
    set c4 [expr {pow($wi,2)/4.0}]
    for {set i $istart} {$i < $iend} {incr i} {
      set square [expr {pow([lindex $xline $i]-$sh,2)}]
      lset zlinetheo $i \
        [expr {$c1*exp($square*$c2)+$c3/($square+$c4)+[lindex $zlinetheo $i]}]
    }
  }
  for {set i 0} {$i < [llength $xline]} {incr i} {
    set xi2 [expr {pow([lindex $zline $i]-[lindex $zlinetheo $i],2)+$xi2}]
  }
  return $xi2
}

proc updatesum {} {
  global xliner pi npeaks peakpar currentpeak fitpar fact znoise
  set peakpar(shift$currentpeak(peak)) $currentpeak(shift)
  set peakpar(area$currentpeak(peak)) $currentpeak(area)
  set peakpar(width$currentpeak(peak)) $currentpeak(width)
  set peakpar(gl$currentpeak(peak)) $currentpeak(gl)
  set f2llocal [lindex $xliner 0]
  set f2rlocal [lindex $xliner end]
  set ncollocal [llength $xliner]
  set zlinesum "0.0[string repeat " 0.0" [expr {[llength $xliner]-1}]]"
  for {set j 0} {$j < $npeaks} {incr j} {
    set sh [callpeakpar shift$j [array get peakpar]] 
    set ar [callpeakpar area$j [array get peakpar]] 
    set wi [callpeakpar width$j [array get peakpar]] 
    set gl [callpeakpar gl$j [array get peakpar]] 
    if {$wi == 0.0} {
      set wi 0.0000000001
    }
#evaluate lineshape
#determine startindex
#determine endindex
   if {[expr abs($ar)*$wi/($pi*2.0*$znoise*$fact)-pow($wi,2)/4.0 > 0.0]} {
     set lorest [expr {sqrt(abs($ar)*$wi/($pi*2.0*$znoise*$fact)-pow($wi,2)/4.0)}]
     set gauest [expr {sqrt(abs(-log(abs($ar)*$wi*sqrt($pi/log(2.0))/(2.0*$znoise*$fact))*pow($wi,2.0)/4.0/log(2.0)))}]
   } else { 
     set lorest [expr {15.0*$wi}]
     set gauest [expr {15.0*$wi}]
   }   
    set est [expr {$gl*$gauest+(1.0-$gl)*$lorest}]
    set est [expr $est < $wi*2.0 ? $wi*2.0 : $est]
    if {[catch {set istart [expr {int(($f2llocal-($sh+$est))/($f2llocal-$f2rlocal)*double($ncollocal))}]}] || $istart < 0} {
      set istart 0
    }
    if {[catch {set iend [expr {int(($f2llocal-($sh-$est))/($f2llocal-$f2rlocal)*double($ncollocal))}]}] || $iend > $ncollocal} {
      set iend $ncollocal
    }
    if {$istart > $iend} {
      set istart 0
      set iend $ncollocal
    }
#some constants c1,...
    set c1 [expr {$ar*$gl*2.0/$wi*sqrt(log(2.0)/$pi)}]
    set c2 [expr {-4.0*log(2.0)/pow($wi,2)}]
    set c3 [expr {$ar*(1.0-$gl)*0.5/$pi*$wi}]
    set c4 [expr {pow($wi,2)/4.0}]
    for {set i $istart} {$i < $iend} {incr i} {
      set square [expr {pow([lindex $xliner $i]-$sh,2)}]
      lset zlinesum $i \
        [expr {$c1*exp($square*$c2)+$c3/($square+$c4)+[lindex $zlinesum $i]}]
    }
  }
  .rf.mg element configure linesum \
    -xdata $xliner \
    -ydata $zlinesum
  set fitpar [list ]
  .lf.fitpar.xi2 configure -text "xi2: [xi2 [array get peakpar]]"
}

proc full {} {
  global f2l f2r xmin xmax
  set xmin $f2r
  set xmax $f2l
  set ok [expand1D]
  if {[.rf.mg element names line1peak]=="line1peak"} {
    updatesinglepeak
    updatesum
  }
}

proc hlp_message {args} {
    set width 600
    #
    # Set up the help window
    toplevel .hlp
    grab .hlp
    wm title .hlp "Help"
    wm minsize .hlp 600 300
    set fontbold "helvetica 10 bold"
    set fontnormal "helvetica 10 normal"
    #
    # Create the window areas
    frame .hlp.top -relief raised -bd 1 
    pack .hlp.top -side top -fill both
    frame .hlp.bot -relief raised -bd 1
    pack .hlp.bot -side bottom -fill both
    #
    # Fill the top part with the message and the bitmap
    set i 0
    label .hlp.top.bitmap -bitmap ""
    pack .hlp.top.bitmap -side left -padx 3m -pady 3m
    foreach text $args {
        if {[expr {fmod($i,2)}]} {
            message .hlp.top.msg$i -width $width -text $text -justify left\
                    -font $fontnormal -anchor w
        } else {
            message .hlp.top.msg$i -width $width -text $text \
                    -font $fontbold
        }
        pack .hlp.top.msg$i -side top -fill x -padx 1m
        incr i
    }
    #
    # Create an exit button
    button .hlp.bot.exit -text Done -command {
        destroy .hlp
    }
    pack .hlp.bot.exit -padx 3m -pady 3m -fill both
    tkwait window .hlp
}

proc updaterow {} {
  global zmatrix rowi zline zlinesq xi2norm znoise phase
  set xi2norm 1.0e-13
  if {[llength [lindex [lindex $zmatrix $rowi] 0]] == 1} {
    set zline [lindex $zmatrix $rowi]
  } else {
    set zline ""
    set sin [expr {sin($phase/57.29577951308)}]
    set cos [expr {cos($phase/57.29577951308)}]
    foreach zre [lindex [lindex $zmatrix $rowi] 0] zim [lindex [lindex $zmatrix $rowi] 1] {
      lappend zline [expr {$cos*$zre+$sin*$zim}]
    }
  }
#calculate xi2norm
  set zlinesq ""
  set zmaxsq -1.0E88
  foreach x $zline {
    set sq [expr {pow($x,2)}]
    set xi2norm [expr {$sq+$xi2norm}]
    lappend zlinesq $sq
    if {$zmaxsq < $sq} {
      set zmaxsq $sq
    }
  }
#determine znoise
#sort squared intensities, take smallest 20%
  set rms20 [lindex [lsort -real $zlinesq] [expr {[llength $zline]/5}]]
  set znoise [expr {4.0*sqrt($rms20)}]
  if {$znoise > 0.0} {
    puts "noise/signal=[expr {4.0*sqrt($rms20/$zmaxsq)}]; noise=[expr {4.0*sqrt($rms20)}]"
  } else {
    set znoise 1.0E-99
  }
  return [update1D]
}

proc update1D {} {
  global zline xline zliner xliner xmin xmax pores ncol f2l f2r
  if {[expr abs($xmax-$xmin) <= ($f2l-$f2r)/($ncol-1)]} {
    set xmin $f2r
    set xmax $f2l
  }
  set xrange [expr {$xmax-$xmin}]
  set nopoints [expr {int($ncol*$xrange/($f2l-$f2r))}]
  set avepo [expr {int($nopoints/$pores)+1}]
  set xliner ""
  set zliner ""
  set xav 0.0
  set zav 0.0
  set actualave 0
  set istart [expr {int(($f2l-$xmax)/(($f2l-$f2r)/$ncol))}]
  set iende [expr {int(($f2l-$xmin)/(($f2l-$f2r)/$ncol))}]
  for {set i $istart} {$i < $iende} {incr i} {
    if {[expr {$i%$avepo+1}] != $avepo} {
      set xav [expr {[lindex $xline $i]+$xav}]
      set zav [expr {[lindex $zline $i]+$zav}]
      incr actualave
    } else {
      set xav [expr {[lindex $xline $i]+$xav}]
      set zav [expr {[lindex $zline $i]+$zav}]
      incr actualave
      lappend xliner [expr {$xav/$actualave}]
      lappend zliner [expr {$zav/$actualave}]
      set xav 0.0
      set zav 0.0
      set actualave 0
    }
  }
  if {[expr {($i-1)%$avepo+1}] != $avepo} {
    lappend xliner [expr {$xav/$actualave}]
    lappend zliner [expr {$zav/$actualave}]
  }
  .rf.mg element configure line1 \
    -xdata $xliner \
    -ydata $zliner
  .rf.mg axis configure x \
    -min $xmin \
    -max $xmax 
  return 1
}

proc expand1D {} {
  global zline xline zliner xliner xmin xmax pores ncol f2l f2r zmax zmin
  if {[expr abs($xmax-$xmin) <= ($f2l-$f2r)/($ncol)]} {
    set xmin $f2r
    set xmax $f2l
  }
  if {$xmin < $f2r} {
    set xmin $f2r
  }
  if {$xmax > $f2l} {
    set xmax $f2l
  }
  set xrange [expr {$xmax-$xmin}]
  set nopoints [expr {int($ncol*$xrange/($f2l-$f2r))}]
  set avepo [expr {int($nopoints/$pores)+1}]
  set xliner ""
  set zliner ""
  set xav 0.0
  set zav 0.0
  set actualave 0
  set istart [expr {int(($f2l-$xmax)*($ncol-1)/($f2l-$f2r))}]
  set iende  [expr {int(($f2l-$xmin)*($ncol-1)/($f2l-$f2r))}]
  set zmax -1.0E88
  set zmin +1.0E88
  for {set i $istart} {$i < $iende} {incr i} {
    if {[expr {$i%$avepo+1}] != $avepo} {
      set xav [expr {[lindex $xline $i]+$xav}]
      set zav [expr {[lindex $zline $i]+$zav}]
      incr actualave
    } else {
      set xav [expr {[lindex $xline $i]+$xav}]
      set zav [expr {[lindex $zline $i]+$zav}]
      incr actualave
      lappend xliner [expr {$xav/$actualave}]
      lappend zliner [expr {$zav/$actualave}]
      set xav 0.0
      set zav 0.0
      set actualave 0
    }
    if {[lindex $zline $i] > $zmax} {
      set zmax [lindex $zline $i]
    } 
    if {[lindex $zline $i] < $zmin} {
      set zmin [lindex $zline $i]
    }
  }
  if {[expr {($i-1)%$avepo+1}] != $avepo} {
    lappend xliner [expr {$xav/$actualave}]
    lappend zliner [expr {$zav/$actualave}]
  }
  .rf.mg element configure line1 \
    -xdata $xliner \
    -ydata $zliner 
  .rf.mg axis configure x \
    -min $xmin \
    -max $xmax 
  if {$zmax <= $zmin} {
    set zmax [expr {pow(1.00000001,$zmin/abs($zmin))*$zmin}]
  }
  .rf.mg axis configure y -max $zmax -min $zmin
  return 1
}

proc bindingschangepeak {} {
  bind .rf.mg <ButtonPress-2> {
   .rf.mg configure -cursor "question_arrow"
    set x1ppm [%W axis invtransform x %x]
    set y1ppm [%W axis invtransform y %y]
    set x1 %x
    set y1 %y
    set iwidthold $currentpeak(width)
    set iglold $currentpeak(gl)
    set iareaold $currentpeak(area)
    set ppmx [%W axis invtransform x %x]
    set amply [lindex $zline [expr {int(($f2l-$ppmx)/($f2l-$f2r)*$ncol)}]]
    .rf.mg marker create text -coords {-Inf Inf} -justify right -anchor ne -text "delta: [format "%%8.2f" $ppmx] ppm\nint.: [format "%%8.4e" $amply]" -outline red -name notification
  }
  bind .rf.mg  <B2-Motion> {
    set x2ppm [%W axis invtransform x %x]
    set y2ppm [%W axis invtransform y %y]
    if {[expr abs(%x-$x1)] > [expr abs(%y-$y1)]} {
      if {[string is double $currentpeak(width)] && [string is double $currentpeak(area)]} {
        .rf.mg configure -cursor "sb_h_double_arrow"
        set currentpeak(width) [expr {$iwidthold*(1.0+($x1ppm-$x2ppm)/($xmax-$xmin))}]
        set currentpeak(area) [expr {$iareaold*(1.0+($x1ppm-$x2ppm)/($xmax-$xmin))}]
        .lf.onepeak.width.txt configure -bg LightYellow
        .lf.onepeak.width.sb configure -bg LightYellow
        .lf.onepeak.gl.txt configure -bg grey85
        .lf.onepeak.gl.sb configure -bg grey85
      } else {
        set currentpeak(area) $peakpar(area$currentpeak(peak))
        set currentpeak(width) $peakpar(width$currentpeak(peak))
      }
    } else { 
      if {[string is double $currentpeak(gl)]} {
        .rf.mg configure -cursor "sb_v_double_arrow"
        set currentpeak(gl) [expr {$iglold+($y2ppm-$y1ppm)/($zmax-$zmin)/2.0}]
        .lf.onepeak.width.txt configure -bg grey85
        .lf.onepeak.width.sb configure -bg grey85
        .lf.onepeak.gl.txt configure -bg LightYellow
        .lf.onepeak.gl.sb configure -bg LightYellow
      } else {
        set currentpeak(area) $peakpar(gl$currentpeak(peak))        
      }
    }
    updatesinglepeak
  }
  bind .rf.mg <ButtonRelease-2> {
    .lf.onepeak.width.txt configure -bg grey85
    .lf.onepeak.width.sb configure -bg grey85
    .lf.onepeak.gl.txt configure -bg grey85
    .lf.onepeak.gl.sb configure -bg grey85
    set x2ppm [%W axis invtransform x %x]
    set y2ppm [%W axis invtransform y %y]
    if {[expr abs(%x-$x1)] > [expr abs(%y-$y1)]} {
      if {[string is double $currentpeak(width)] && [string is double $currentpeak(area)]} {
        set currentpeak(width) [expr $iwidthold*(1.0+($x1ppm-$x2ppm)/($xmax-$xmin))]
        if {$currentpeak(width) < 0.00000001} {
          set currentpeak(width) 0.00000001
        }
        set currentpeak(area) [expr {$iareaold*$currentpeak(width)/$iwidthold}]
      } else {
        set currentpeak(area) $peakpar(area$currentpeak(peak))
        set currentpeak(width) $peakpar(width$currentpeak(peak))
      }      
    } else { 
      if {[string is double $currentpeak(gl)]} {
        set currentpeak(gl) [expr {$iglold+($y2ppm-$y1ppm)/($zmax-$zmin)/2.0}]
        if {$currentpeak(gl) > 1.0} {
          set currentpeak(gl) 1.0
        } elseif {$currentpeak(gl) < 0.0} {
          set currentpeak(gl) 0.0
        }
      } else {
        set currentpeak(area) $peakpar(gl$currentpeak(peak))        
      }
    }
    updatesinglepeak
    updatesum
    .rf.mg marker delete notification
    .rf.mg configure -cursor "crosshair"
  }
  bind .rf.mg <ButtonPress-3> {
    .rf.mg configure -cursor "question_arrow"
    set x1ppm [%W axis invtransform x %x]
    set y1ppm [%W axis invtransform y %y]
    set x1 %x
    set y1 %y
    set ishiftold $currentpeak(shift)
    set iareaold $currentpeak(area)
    set ppmx [%W axis invtransform x %x]
    set amply [lindex $zline [expr {int(($f2l-$ppmx)/($f2l-$f2r)*$ncol)}]]
    .rf.mg marker create text -coords {-Inf Inf} -justify right -anchor ne -text "delta: [format "%%8.2f" $ppmx] ppm\nint.: [format "%%8.4e" $amply]" -outline red -name notification
  }
  bind .rf.mg  <B3-Motion> {
    set x2ppm [%W axis invtransform x %x]
    set y2ppm [%W axis invtransform y %y]
    if {[expr abs(%x-$x1)] > [expr abs(%y-$y1)]} {
      if {[string is double $currentpeak(shift)]} {
        .rf.mg configure -cursor "sb_h_double_arrow"
        .lf.onepeak.shift.txt configure -bg LightYellow
        .lf.onepeak.shift.sb configure -bg LightYellow
        .lf.onepeak.area.txt configure -bg grey85
        .lf.onepeak.area.sb configure -bg grey85
        set currentpeak(shift) [expr $ishiftold+$x2ppm-$x1ppm]
      } else {
        set currentpeak(shift) $peakpar(shift$currentpeak(peak))        
      }
    } else { 
      if {[string is double $currentpeak(area)]} {
        .rf.mg configure -cursor "sb_v_double_arrow"
        set currentpeak(area) [expr $iareaold*(1.0+($y2ppm-$y1ppm)/($zmax-$zmin))]
        .lf.onepeak.shift.txt configure -bg grey85
        .lf.onepeak.shift.sb configure -bg grey85
        .lf.onepeak.area.txt configure -bg LightYellow
        .lf.onepeak.area.sb configure -bg LightYellow
      } else {
        set currentpeak(area) $peakpar(area$currentpeak(peak))        
      }
    }
    updatesinglepeak
  }
  bind .rf.mg <ButtonRelease-3> {
    .lf.onepeak.shift.txt configure -bg grey85
    .lf.onepeak.shift.sb configure -bg grey85
    .lf.onepeak.area.txt configure -bg grey85
    .lf.onepeak.area.sb configure -bg grey85
    set x2ppm [%W axis invtransform x %x]
    set y2ppm [%W axis invtransform y %y]
    if {[expr abs(%x-$x1)] > [expr abs(%y-$y1)]} {
      if {[string is double $currentpeak(shift)]} {
        set currentpeak(shift) [expr $ishiftold+$x2ppm-$x1ppm]
      } else {
        set currentpeak(shift) $peakpar(shift$currentpeak(peak))        
      }
    } else {
      if {[string is double $currentpeak(area)]} {
        set currentpeak(area) [expr $iareaold*(1.0+($y2ppm-$y1ppm)/($zmax-$zmin))]
      } else {
        set currentpeak(area) $peakpar(area$currentpeak(peak))        
      }
    }
    .rf.mg marker delete notification
    updatesinglepeak
    updatesum
    .rf.mg configure -cursor "crosshair"
  }
}

#mousebindings to expand 1D
proc bindingsexpand {} {
    bind .rf.mg <ButtonPress-1> {
    set x1 [%W axis invtransform x %x]
    set y1 [lindex $zline [expr {int(($f2l-$x1)/($f2l-$f2r)*$ncol)}]]
    .rf.mg configure -cursor "sb_h_double_arrow"
    .rf.mg marker create line -coords {$x1 $zmin $x1 $zmax $x1 $zmax $x1 $zmin $x1 $zmin} -name expregion \
      -dashes dash -xor yes
    .rf.mg marker create text -coords {-Inf Inf} -justify right -anchor ne -text "delta: [format "%%8.2f" $x1] ppm\nint.: [format "%%8.4e" $y1]" -outline red -name notificationwidth
  }
  bind .rf.mg  <B1-Motion> {
    set x2 [%W axis invtransform x %x]
      .rf.mg marker create text -coords {-Inf Inf} -justify right -anchor ne -text "left: [format "%%8.2f" $x1] ppm\nright: [format "%%8.2f" $x2] ppm\ndelta: [format "%%8.2f" [expr {$x1-$x2}]] ppm" -outline red -name notificationwidth
    .rf.mg marker configure expregion -coords {$x1 $zmin $x1 $zmax $x2 $zmax $x2 $zmin $x1 $zmin}
  }
  bind .rf.mg <ButtonRelease-1> {
    .rf.mg marker delete notificationwidth
    set x2 [%W axis invtransform x %x]
    .rf.mg marker delete expregion
    .rf.mg configure -cursor "crosshair"
    if {$x1 < $x2} {
      set xmin $x1
      set xmax $x2
    } else {
      set xmin $x2
      set xmax $x1
    }
    set ok [expand1D]
    if {[.rf.mg element names line1peak]=="line1peak"} {
      updatesinglepeak
      updatesum
    }
  }
}

proc loadlineshape {} {
  global npeaks peakpar boolfit currentpeak xmin xmax nrow pi currentpeak
  set filetypes [list {deconv2Dxy .log1D}]
  set filename [tk_getOpenFile -title "open old fit" -filetypes $filetypes]
  if {$filename == ""} {
  } else {
    set rawfile [readrawfile $filename]
#map names
    set nonewpeaks 0
    foreach line $rawfile {
      if {[string index $line 0] != "#" && [string index $line 0] != ""} {
        incr nonewpeaks
      }
    }
    set stringmap ""
    for {set i [expr {$nonewpeaks-1}]} {$i > -1} {incr i -1} {
      lappend stringmap shift$i shift[expr {$npeaks+$i}]
      lappend stringmap area$i area[expr {$npeaks+$i}]
      lappend stringmap width$i width[expr {$npeaks+$i}]
      lappend stringmap gl$i gl[expr {$npeaks+$i}]
    }
    set i 0
    foreach line $rawfile {
      if {[string index $line 0] != "#" && [string index $line 0] != ""} {
        set splittedline [split $line]
        set peakpar([string map $stringmap shift$i]) [string map $stringmap [lindex $splittedline 1]]
        if {[string is double $peakpar([string map $stringmap shift$i])]} {
          set boolfit([string map $stringmap shift$i]) 1
        } else {
          set boolfit([string map $stringmap shift$i]) 0
        }
        set peakpar([string map $stringmap area$i]) [string map $stringmap [lindex $splittedline 2]]
        if {[string is double $peakpar([string map $stringmap area$i])]} {
          set boolfit([string map $stringmap area$i]) 1
        } else {
          set boolfit([string map $stringmap area$i]) 0
        }
        set peakpar([string map $stringmap width$i]) [string map $stringmap [lindex $splittedline 3]]
        if {[string is double $peakpar([string map $stringmap width$i])]} {
          set boolfit([string map $stringmap width$i]) 1
        } else {
          set boolfit([string map $stringmap width$i]) 0
        }
        set peakpar([string map $stringmap gl$i]) [string map $stringmap [lindex $splittedline 4]]
        set boolfit([string map $stringmap gl$i]) 0
        incr npeaks
        incr i
      }
    }
    if {$npeaks > 0} {
      bindingschangepeak
    }
    if {$npeaks > $nonewpeaks} {
      updatesinglepeak
      updatesum
      .lf.general.totalpeaks configure -text "total no. peaks: [expr $npeaks]"
      .lf.onepeak.peak.sb configure -from 0 -to [expr $npeaks-1]
    } elseif {$npeaks == $nonewpeaks} {
      copypeakpar 0 current
      set currentpeak(oldpeak) 0
      .lf.general.totalpeaks configure -text "total no. peaks: [expr $npeaks]"
      .lf.onepeak.peak.sb configure -from 0 -to [expr $npeaks-1]
      .rf.mg element create line1peak\
      -xdata {$xmin $xmax} \
      -ydata {0.0 0.0}
      .rf.mg element create linesum\
      -xdata {$xmin $xmax} \
      -ydata {0.0 0.0}
      .rf.mg element configure line1peak -symbol no -color blue
      .rf.mg element configure linesum -symbol no -color red
      pack .lf.general.ar.remove
      if {$nrow==1} {
        pack .lf.onepeak.gl .lf.onepeak.width .lf.onepeak.area .lf.onepeak.shift .lf.onepeak.peak -in .lf.onepeak -side bottom -expand yes -fill x
        pack .lf.fitpar.fit1D .lf.fitpar.fit1Damp .lf.fitpar.fit1Dcur .lf.fitpar.xi2 -in .lf.fitpar -side bottom -fill x
        #.lf.fitpar.us
      } else {
        pack .lf.onepeak.gl .lf.onepeak.width .lf.onepeak.area .lf.onepeak.shift .lf.onepeak.peak -in .lf.onepeak -side bottom -expand yes -fill x
        pack .lf.fitpar.fit1D .lf.fitpar.fit1Damp .lf.fitpar.fit1Dcur .lf.fitpar.fit2D .lf.fitpar.xi2 -in .lf.fitpar -side bottom -fill x
        #.lf.fitpar.us
      }
      updatesinglepeak
      updatesum
    }
  }
}

#needed to format tick labels for y-axis
proc double2float {pathname value} {
  return [format "%g" $value]
}

proc loadfile {} {
  global filename
  set filetypes [list {Bruker .txt} {SIMPSON .spe}]
  set filename [tk_getOpenFile -title "open NMR spectrum" -filetypes $filetypes]
  set filenameroot [string range $filename 0 end-4]
  set fileextension [string range $filename end-2 end]
  if {$filenameroot == ""} {
  } else {
    readandsetup $filenameroot $fileextension
  }
}

proc readandsetup {filename fileextension} {
  global zmin zmax xline zlinetheo zmatrix nrow ncol f2l f2r xmin xmax npeaks rowi peakpar currentpeak version reffreq
  pack forget .lf.general.ar .lf.general.totalpeaks .lf.general.fs .lf.general.row
  wm title . "deconv2Dxy-$version - file: $filename"
  set rowi 0
  set nrow 0
  set lastraw -1
#read 2D spectra
#determine smallest and biggest z element
  set zmin 1.0E+88
  set zmax -1.0E+88
  set xline ""
  set zlinetheo ""
  if {$fileextension == "txt"} {
    set zmatrix [readdatabruker $filename.txt]
  } elseif {$fileextension == "spe"} {
    set zmatrix [readdatasimpson $filename.spe]
  }
  if {[llength [lindex [lindex $zmatrix 0] 0]] == 1} {
    pack forget .lf.general.ph
    for {set i 0} {$i < $nrow} {incr i} {
      set zmaxrow -1.0e+88
      for {set j 0} {$j < $ncol} {incr j} {
        set z [lindex [lindex $zmatrix $i] $j]
        if {$z < $zmin} {
          set zmin $z
        }
        if {$z > $zmax} {
          set zmax $z
        }
        if {$z > $zmaxrow} {
          set zmaxrow $z
        }
      }
#determine last non-zero row 
      if {$zmaxrow != 0.0} {
        set nrowwithint $i
      }
    }
  } else {
    pack forget .lf.general.ph
    pack .lf.general.ph
    pack .lf.general.scale
    for {set i 0} {$i < $nrow} {incr i} {
      set zmaxrow -1.0e+88
      for {set j 0} {$j < $ncol} {incr j} {
        set z [lindex [lindex [lindex $zmatrix $i] 0] $j]
        if {$z < $zmin} {
          set zmin $z
        }
        if {$z > $zmax} {
          set zmax $z
        }
        if {$z > $zmaxrow} {
          set zmaxrow $z
        }
      }
#determine last non-zero row 
      if {$zmaxrow != 0.0} {
        set nrowwithint $i
      }
    }
  }
  set nrow [expr {$nrowwithint+1}]
  if {$ncol > 4096} {
    set xfilereply [tk_messageBox -message "You try to fit a spectrum with $ncol number of points. With more than 4K points the program requires lots of patience." \
        -type ok]
  }
  for {set j 0} {$j < $ncol} {incr j} {
    lappend xline [expr {($f2r-$f2l)*$j/($ncol-1)+$f2l}]
    lappend zlinetheo 0.0
  }
  set xmin $f2r
  set xmax $f2l
  .rf.mg axis configure y -min $zmin -max $zmax
  .rf.mg legend configure -hide yes
  .lf.general.row.sb configure -textvariable rowi -from 0 -to [expr $nrow-1]
  .lf.onepeak.width.sb configure -from 0.0 -to [expr $f2l-$f2r] -increment [expr abs($f2l-$f2r)/50.0] 
  .lf.onepeak.shift.sb configure -from $f2r -to $f2l -increment [expr ($xmax-$xmin)/10.0]
  .lf.onepeak.area.sb configure -increment [expr $zmax/20.0] -from [expr $zmin*($xmax-$xmin)] -to [expr $zmax*($xmax-$xmin)]
  set ok [updaterow]
  if {[.rf.mg element names line1peak]=="line1peak"} {
    set currentpeak(shift) $peakpar(shift$currentpeak(peak))
    set currentpeak(area) $peakpar(area$currentpeak(peak))
    set currentpeak(width) $peakpar(width$currentpeak(peak))
    set currentpeak(gl) $peakpar(gl$currentpeak(peak))
    updatesinglepeak
    updatesum
  }
#add appropriate widgets if 2D or 1D
  if {$nrow==1} {
    pack forget .lf.fitpar.fit2D
  } elseif {$npeaks > 0} {
    pack .lf.fitpar.fit2D -in .lf.fitpar -side bottom -fill x
  }
  if {$nrow==1} {
    pack forget .lf.general.row.sb .lf.general.row.txt
    pack .lf.general.totalpeaks .lf.general.fs .lf.general.ar -in .lf.general -fill x -expand yes -side bottom 
  } else {
    pack .lf.general.row.sb .lf.general.row.txt -in .lf.general.row -side bottom
    pack .lf.general.totalpeaks .lf.general.fs .lf.general.ar .lf.general.row -in .lf.general -side bottom -fill x -expand yes
  }
  bindingsexpand
}

#read SIMPSON spe file
proc readdatasimpson {filename} {
  global f1l f1r f2l f2r nrow ncol version boolfit reffreq refoffset
  set zmatrix ""
  set zline ""
  set zlinere ""
  set zlineim ""
  set raw [readrawfile $filename]
  set lineindex 0
  for {set i 0} {$i < [llength $raw]} {incr i} {
    if {[string match -nocase "DATA" [lindex $raw $i]]} {
      for {set i $i} {![string match -nocase "END*" [lindex $raw [expr $i+1]]]} {incr i} {
        set line [lindex "\{[split [lindex $raw [expr $i+1]] "\} \{"]\}" 0]     
        set blank [lsearch $line ""]
        while {$blank!=-1} {
          set blank [lsearch $line ""]
          set line [lreplace $line $blank $blank]
        }
        lappend rawdata [list [lindex $line 0] [lindex $line 1]]
        incr lineindex
      }
    } elseif {[string match -nocase "*SW=*" [lindex $raw $i]]} {
      set sw [expr double([string range [lindex $raw $i] 3 end])]
    } elseif {[string match -nocase "# freq. of reference compound (0ppm) in MHz:*" [lindex $raw $i]]} {
      set reffreq [expr {double([string range [lindex $raw $i] 45 end])*1.0e6}]
    } elseif {[string match -nocase "REF=*" [lindex $raw $i]]} {
      set refoffset [expr {double([string range [lindex $raw $i] 4 end])}]
    }
  }
  set ncol [llength $rawdata]
  set f2l [expr ($sw/2.0+$refoffset-$sw/double($ncol))/$reffreq*1.0e6]
  set f2r [expr (-$sw/2.0+$refoffset)/$reffreq*1.0E6]
  set nrow 1
  foreach xy $rawdata {
    lappend zlinere [lindex $xy 0]
    lappend zlineim [lindex $xy 1]
  }
  set zmatrix [list [list [lreverse $zlinere] [lreverse $zlineim]]]
  wm title . "deconv2Dxy-$version - file: $filename"
  return $zmatrix
}

#read BRUKER txt file
proc readdatabruker {filename} {
  global f1l f1r f2l f2r nrow ncol version boolfit
  set zmatrix ""
  set zline ""
  set zlinere ""
  set zlineim ""
  set raw [readrawfile $filename]
  set lineindex 0
  for {set i 0} {$i < [llength $raw]} {incr i} {
    if {[string match "*F1LEFT*" [lindex $raw $i]]} {
      set line [lindex $raw $i]
      set ia [string first "F1LEFT" $line]
      set ib [string first "ppm" $line $ia]
      set f1l [string range $line [expr {$ia+9}] [expr {$ib-2}]]
      set ia [string first "F1RIGHT" $line]
      set ib [string first "ppm" $line $ia]
      set f1r [string range $line [expr {$ia+10}] [expr {$ib-2}]]
      incr i
      set line [lindex $raw $i]
      set ia [string first "F2LEFT" $line]
      set ib [string first "ppm" $line $ia]
      set f2l [string range $line [expr {$ia+9}] [expr {$ib-2}]]
      set ia [string first "F2RIGHT" $line]
      set ib [string first "ppm" $line $ia]
      set f2r [string range $line [expr {$ia+10}] [expr {$ib-2}]]
    } elseif {[string match "*NROWS*" [lindex $raw $i]]} {
      set line [lindex $raw $i] 
      set ia [string first "NROWS" $line]
      set nrow [string range $line [string wordstart $line [expr {$ia+8}]] [string wordend $line [expr {$ia+8}]]]
    } elseif {[string match "*NCOLS*" [lindex $raw $i]]} {
      set line [lindex $raw $i] 
      set ia [string first "NCOLS" $line]
      set ncol [string range $line [string wordstart $line [expr {$ia+8}]] [string wordend $line [expr {$ia+8}]]]
    } elseif {[string match "# LEFT =*" [lindex $raw $i]]} {
      set line [lindex $raw $i] 
      set ia [string first "LEFT" $line]
      set ib [string first "ppm" $line $ia]
      set f2l [string range $line [string wordstart $line [expr {$ia+7}]] [string wordend $line [expr {$ib-2}]]]
      set ia [string first "RIGHT" $line]
      set ib [string first "ppm" $line $ia]
      set f2r [string range $line [string wordstart $line [expr {$ia+8}]] [string wordend $line [expr {$ib-2}]]]
      set nrow 1
      set line [lindex $raw [expr {$i+2}]] 
      set ia [string first "SIZE" $line]
      set ncol [string range $line [string wordstart $line [expr {$ia+7}]] [string wordend $line [expr {$ia+7}]]]
    } elseif {[string match "# row*" [lindex $raw $i]]} {
    } elseif {[string match "#*" [lindex $raw $i]]} {
    } elseif {[string match "*i*" [lindex $raw $i]]} {
      if {[expr {$lineindex % $ncol}]==[expr {$ncol-1}]} {
        set z [string map {E e} [lindex $raw $i]]
        lappend zlinere [complex::re $z]
        lappend zlineim [complex::im $z]
        lappend zmatrix [list $zlinere $zlineim]
        set zline ""
      } else {
        set z [string map {E e} [lindex $raw $i]]
        lappend zlinere [complex::re $z]
        lappend zlineim [complex::im $z]
      }
      incr lineindex
    } else {
      if {[expr {$lineindex % $ncol}]==[expr {$ncol-1}]} {
        set z [lindex $raw $i]
        lappend zline $z
        lappend zmatrix $zline
        set zline ""
      } else {
        set z [lindex $raw $i]
        lappend zline $z
      }
      incr lineindex
    }
  }
#if non-imaginary data set
  if {$zlineim==""} {
    set boolfit(phase) 0
  }
  wm title . "deconv2Dxy-$version - file: $filename"
  return $zmatrix
}

proc default {{wide 6}} {
  global iterationsperpeak epsilon fact verbose
  # Set up the input window
  toplevel .default
  grab .default
  wm title .default "default parameters"
  #max. iterations
  frame .default.iter
  label .default.iter.text -text "iterations per peak      "
  entry .default.iter.ent -width $wide -relief sunken -textvariable iterationsperpeak
  pack  .default.iter.text .default.iter.ent -in .default.iter -padx 2m -pady 2m -side left
  #max. convergence criterion
  frame .default.epsilon
  label .default.epsilon.text -text "convergence criterion "
  entry .default.epsilon.ent -width $wide -relief sunken -textvariable epsilon
  pack  .default.epsilon.text .default.epsilon.ent -in .default.epsilon -padx 2m -pady 2m -side left
  #cut off
  frame .default.fact
  label .default.fact.text -text "cut off factor               "
  entry .default.fact.ent -width $wide -relief sunken -textvariable fact
  pack  .default.fact.text .default.fact.ent -in .default.fact -padx 2m -pady 2m -side left
  #verbosity
  frame .default.verb
  label .default.verb.text -text "verbosity during fit      "
  entry .default.verb.ent -width $wide -relief sunken -textvariable verbose
  pack  .default.verb.text .default.verb.ent -in .default.verb -padx 2m -pady 2m -side left
  #pack
  pack .default.epsilon .default.iter .default.fact .default.verb -in .default -side bottom
  # The routine exits when the value is entered
  bind .default.epsilon.ent <Return> {
    destroy .default
  }
  bind .default.iter.ent <Return> {
    destroy .default
  }
  bind .default.fact.ent <Return> {
    destroy .default
  }
  # Wait until the window is destroyed before completing procedure
  tkwait window .default
}

proc setxscale {{wide 6}} {
  global xmin xmax f2l f2r
  # Set up the input window
  toplevel .xscale
  grab .xscale
  wm title .xscale "frequency scale"
  #right limit
  frame .xscale.left
  label .xscale.left.text -text "left limit/ppm "
  entry .xscale.left.ent -width $wide -relief sunken -textvariable xmin
  pack  .xscale.left.text .xscale.left.ent -in .xscale.left -padx 2m -pady 2m -side left
  #left limit
  frame .xscale.right
  label .xscale.right.text -text "right limit/ppm"
  entry .xscale.right.ent -width $wide -relief sunken -textvariable xmax
  pack  .xscale.right.text .xscale.right.ent -in .xscale.right -padx 2m -pady 2m -side left
  #pack
  pack .xscale.left .xscale.right  -in .xscale -side bottom
  # The routine exits when the value is entered
  bind .xscale.left.ent <Return> {
    destroy .xscale
    update1D
  }
  bind .xscale.right.ent <Return> {
    destroy .xscale
    update1D
  }
  # Wait until the window is destroyed before completing procedure
  tkwait window .xscale
}

proc setyscale {{wide 6}} {
  global zmin zmax
  # Set up the input window
  toplevel .yscale
  grab .yscale
  wm title .yscale "intensity scale"
  #right limit
  frame .yscale.high
  label .yscale.high.text -text "upper limit"
  entry .yscale.high.ent -width $wide -relief sunken -textvariable zmax
  pack  .yscale.high.text .yscale.high.ent -in .yscale.high -padx 2m -pady 2m -side left
  #left limit
  frame .yscale.low
  label .yscale.low.text -text "lower limit"
  entry .yscale.low.ent -width $wide -relief sunken -textvariable zmin
  pack  .yscale.low.text .yscale.low.ent -in .yscale.low -padx 2m -pady 2m -side left
  #pack
  pack .yscale.low .yscale.high -in .yscale -side bottom
  # The routine exits when the value is entered
  bind .yscale.high.ent <Return> {
    destroy .yscale
   .rf.mg axis configure y -max $zmax -min $zmin
  }
  bind .yscale.low.ent <Return> {
    destroy .yscale
   .rf.mg axis configure y -max $zmax -min $zmin
  }
  # Wait until the window is destroyed before completing procedure
  tkwait window .yscale
}

#initialize canvas
frame .menu -relief raised -borderwidth 2
frame .lf 
frame .rf 

frame .lf.onepeak -relief raised -bd 3
frame .lf.fitpar -relief raised -bd 3
frame .lf.general -relief raised -bd 3

frame .lf.onepeak.peak
frame .lf.onepeak.area
frame .lf.onepeak.width
frame .lf.onepeak.gl
frame .lf.onepeak.shift
frame .lf.general.ar
frame .lf.general.row
frame .lf.general.ph
frame .lf.general.scale

#main graph
blt::graph .rf.mg -background white -plotbackground white -width $canvaswidthx -height $canvaswidthy
.rf.mg element create line1 \
    -xdata {0.0 1.0} \
    -ydata {0.0 0.0} -symbol no -color black
.rf.mg legend configure -hide yes
.rf.mg axis configure x -title "chemical shift/ppm" -titlefont *-Helvetica-Bold-R-Normal-*-18-140-* -tickfont *-Helvetica-Bold-R-Normal-*-14-140-* -subdivisions 5 -descending 1
.rf.mg axis configure y -title "intensity/a.u." -titlefont *-Helvetica-Bold-R-Normal-*-18-140-* -tickfont *-Helvetica-Bold-R-Normal-*-14-140-* -subdivisions 5 -command double2float
label .lf.general.totalpeaks -text "total no. peaks: 0"

label .lf.general.row.txt -text "row" 
spinbox .lf.general.row.sb -increment 1 -command updaterow -width 5
bind .lf.general.row.sb <Return> {
  if {[string is integer $rowi] && $rowi < $nrow && $rowi >= 0} {
    updaterow
  } else {
    set rowi 0
  }
}

label .lf.onepeak.peak.txt -text "peak"
spinbox .lf.onepeak.peak.sb -textvariable currentpeak(peak) -increment 1 -command {changeactivepeak %d} -width 3 -wrap 1


label .lf.onepeak.shift.txt -text "shift/ppm"
checkbutton .lf.onepeak.shift.but -variable boolfit(shift) -text "fit"
spinbox .lf.onepeak.shift.sb -textvariable currentpeak(shift) -command updatesinglepeak -increment 0.05 -from 0.0 -to 1.0 
bind .lf.onepeak.shift.sb <Leave> {
  array set testpeakpar [array get peakpar]
  set testpeakpar(shift$currentpeak(peak)) $currentpeak(shift)
  if {[catch {set result [callpeakpar shift$currentpeak(peak) [array get testpeakpar]]}]} {
    set currentpeak(shift) $peakpar(shift$currentpeak(peak))
  } else {
    set peakpar(shift$currentpeak(peak)) $currentpeak(shift)
    if {[string is double $currentpeak(shift)] == 0} {
      set boolfit(shift) 0
    }
  }
  updatesinglepeak
  updatesum
}
bind .lf.onepeak.shift.sb <Return> {
  array set testpeakpar [array get peakpar]
  set testpeakpar(shift$currentpeak(peak)) $currentpeak(shift)
  if {[catch {set result [callpeakpar shift$currentpeak(peak) [array get testpeakpar]]}]} {
    set currentpeak(shift) $peakpar(shift$currentpeak(peak))
  } else {
    set peakpar(shift$currentpeak(peak)) $currentpeak(shift)
    if {[string is double $currentpeak(shift)] == 0} {
      set boolfit(shift) 0
    }
  }
  updatesinglepeak
  updatesum
}

label .lf.onepeak.area.txt -text "area/a.u."
checkbutton .lf.onepeak.area.but -variable boolfit(area) -text "fit"
spinbox .lf.onepeak.area.sb -textvariable currentpeak(area) -command updatesinglepeak -increment 0.05 -from 0.0 -to 1.0 

bind .lf.onepeak.area.sb <Leave> {
  array set testpeakpar [array get peakpar]
  set testpeakpar(area$currentpeak(peak)) $currentpeak(area)
  if {[catch {set result [callpeakpar area$currentpeak(peak) [array get testpeakpar]]}]} {
    set currentpeak(area) $peakpar(area$currentpeak(peak))
  } else {
    set peakpar(area$currentpeak(peak)) $currentpeak(area)
    if {[string is double $currentpeak(area)] == 0} {
      set boolfit(area) 0
    }
  }
  updatesinglepeak
  updatesum
}
bind .lf.onepeak.area.sb <Return> {
  array set testpeakpar [array get peakpar]
  set testpeakpar(area$currentpeak(peak)) $currentpeak(area)
  if {[catch {set result [callpeakpar area$currentpeak(peak) [array get testpeakpar]]}]} {
    set currentpeak(area) $peakpar(area$currentpeak(peak))
  } else {
    set peakpar(area$currentpeak(peak)) $currentpeak(area)
    if {[string is double $currentpeak(area)] == 0} {
      set boolfit(area) 0
    }
  }
  updatesinglepeak
  updatesum
}

label .lf.onepeak.gl.txt -text "GL ratio"
checkbutton .lf.onepeak.gl.but -variable boolfit(gl) -text "fit"
spinbox .lf.onepeak.gl.sb -textvariable currentpeak(gl) -from 0.0 -to 1.0 -increment 0.05 -command updatesinglepeak

bind .lf.onepeak.gl.sb <Leave> {
  array set testpeakpar [array get peakpar]
  set testpeakpar(gl$currentpeak(peak)) $currentpeak(gl)
  if {[catch {set result [callpeakpar gl$currentpeak(peak) [array get testpeakpar]]}]} {
    set currentpeak(gl) $peakpar(gl$currentpeak(peak))
  } else {
    set peakpar(gl$currentpeak(peak)) $currentpeak(gl)
    if {[string is double $currentpeak(gl)] == 0} {
      set boolfit(gl) 0
    }
  }
  updatesinglepeak
  updatesum
}

bind .lf.onepeak.gl.sb <Return> {
  array set testpeakpar [array get peakpar]
  set testpeakpar(gl$currentpeak(peak)) $currentpeak(gl)
  if {[catch {set result [callpeakpar gl$currentpeak(peak) [array get testpeakpar]]}]} {
    set currentpeak(gl) $peakpar(gl$currentpeak(peak))
  } else {
    set peakpar(gl$currentpeak(peak)) $currentpeak(gl)
    if {[string is double $currentpeak(gl)] == 0} {
      set boolfit(gl) 0
    }
  }
  updatesinglepeak
  updatesum
}

label .lf.onepeak.width.txt -text "width/ppm"
checkbutton .lf.onepeak.width.but -variable boolfit(width) -text "fit"
spinbox .lf.onepeak.width.sb -command updatesinglepeak -textvariable currentpeak(width) 

bind .lf.onepeak.width.sb <Leave> {
  array set testpeakpar [array get peakpar]
  set testpeakpar(width$currentpeak(peak)) $currentpeak(width)
  if {[catch {set result [callpeakpar width$currentpeak(peak) [array get testpeakpar]]}]} {
    set currentpeak(width) $peakpar(width$currentpeak(peak))
  } else {
    set peakpar(width$currentpeak(peak)) $currentpeak(width)
    if {[string is double $currentpeak(width)] == 0} {
      set boolfit(width) 0
    }
  }
  updatesinglepeak
  updatesum
}
bind .lf.onepeak.width.sb <Return> {
  array set testpeakpar [array get peakpar]
  set testpeakpar(width$currentpeak(peak)) $currentpeak(width)
  if {[catch {set result [callpeakpar width$currentpeak(peak) [array get testpeakpar]]}]} {
    set currentpeak(width) $peakpar(width$currentpeak(peak))
  } else {
    set peakpar(width$currentpeak(peak)) $currentpeak(width)
    if {[string is double $currentpeak(width)] == 0} {
      set boolfit(width) 0
    }
  }
  updatesinglepeak
  updatesum
}

label .lf.fitpar.xi2 -text "xi2: ?"

menubutton .menu.help -text "?" -menu .menu.help.m -underline 0
menubutton .menu.file -text "file" -menu .menu.file.m -underline 0
menubutton .menu.view -text "view" -menu .menu.view.m -underline 0
menu .menu.help.m -tearoff 0
menu .menu.file.m -tearoff 0
menu .menu.view.m -tearoff 0
.menu.help.m add command -label "help" -command {
    hlp_message {Deconv2Dxy} \
            {This program allows a deconvolution of 1D spectra and stacked 1D spectra packed into a 2D. Its primary use is analysis of 1D spectra and deconvolution of stacked 1D spectra with fixed line-shape and variable intensity parameters. We use it for analysis of experimental data for example from REDOR, DQ-CT and T1-experiments.}\
            {File formats} {So far files exported from Bruker's program "topspin-2/3" and the simulation program SIMPSON may be loaded. Use the command "totxt" from within topspin. If you want to apply a first order phase correction, you need to include imaginary data.}\
            {Usage} {The program is written in TCL/TK alone and is a first-times-tk hack. Still it can easily fit 1-20 peaks given the spectrum size in points is small (<1K). If you are patient you can fit big spectra as well. The parameter max.iterations in "defaults" gives the number of iterations per fitted peak after which the program will stop fitting and warn with a red fit1D-button if convergence had not been reached before. If you have to press the fit button several times to reach convergence, you might consider to increase the value for "max.iterations". It is also possible to change the 0th order phase of a spectrum given you have a data set with complex numbers. Phase fitting is possible but will not always give stable results, especially if the spectrum is noisy or has a distorted baseline. An option for fitting is to use constraints. Any expression of the type shift0*gl0-area1+sqrt(3.0) can be typed into the parameter fields, where shift0 refers to the shift value of peak 0, etc.. The phase parameter for complex data sets is not available however.}\
            {Results} {Peak areas ar, chemical shift values x0, line widths wi (FWHH) and gauss/lorentz ratios gl are put into files with the extension log1D and log2D, respectively. Start the program from the command line to get more feedback.}\
            {Lineshape function} {lineshape = ar*gl*2/wi*sqrt(ln(2)/pi)*exp(-4*ln(2)/wi/wi*(x-x0)^2)+ar*(1-gl)*0.5/pi*wi/(wi^2/4+(x-x0)^2)} \
            {Note} {Mental support in form of gummy bears and friendly emails is always welcome.}\
            {copyright (C) 2008-2018 Joern Schmedt auf der Guenne} {
This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have received a copy of the GNU General Public License along with this program; if not, see <http://www.gnu.org/licenses/>.}\
}
.menu.file.m add command -label "load spectrum" -command loadfile
.menu.file.m add command -label "load fitted lineshape" -command loadlineshape
.menu.file.m add command -label "plot spectrum" -command {printplot $filename}
.menu.file.m add command -label "default parameters" -command default
.menu.file.m add command -label "exit program" -command {exit 0}

.menu.view.m add command -label "frequency limits" -command {setxscale 8}
.menu.view.m add command -label "intensity limits" -command {setyscale 8}

pack .menu.help -side right
pack .menu.file -side left
pack .menu.view -side left
tk_menuBar .menu .menu.file .menu.help

button .lf.general.fs -command full -text "full" -state normal
button .lf.fitpar.fit1D -command fit1D -text "fit 1D lineshape" -state normal
button .lf.fitpar.fit1Damp -command fit1Damp -text "fit areas in 1D" -state normal
button .lf.fitpar.fit1Dcur -command fit1Dcur -text "fit current peak" -state normal
button .lf.fitpar.fit2D -command fit2D -text "fit areas in 2D" -state normal
button .lf.general.ar.add -command addpeak -text "add\npeak" -state normal
button .lf.general.ar.remove -command removepeak -text "remove\npeak" -state normal

label .lf.general.scale.txt -text "reference freq./MHz"
entry .lf.general.scale.en -textvariable reffreq -relief sunken

label .lf.general.ph.txt -text "phase/degree"
checkbutton .lf.general.ph.but -variable boolfit(phase) -text "fit"
spinbox .lf.general.ph.sb -textvariable phase -from -180.0 -to 180.0 -increment 0.1 -command updaterow
wm title . "deconv2Dxy-$version"
bind .lf.general.ph.sb <Leave> {
  if {[.rf.mg element names line1peak]=="line1peak"} {
    if {[string is double $phase]} {
      updatesinglepeak
      updatesum
    } else {
      set phase 0.0
    }
  }
}

pack .menu -side top -fill x
pack .rf -side right -expand yes
pack .lf -side left -expand yes
pack .rf.mg -in .rf -expand yes

pack .lf.onepeak .lf.fitpar .lf.general -in .lf -side top -expand yes -fill x 

pack .lf.general.ar.add   -in .lf.general.ar    -side right -expand yes
pack .lf.general.row.sb   .lf.general.row.txt   -in .lf.general.row   -side bottom

pack .lf.general.ph.sb                          -in .lf.general.ph    -side bottom
pack .lf.general.ph.txt   .lf.general.ph.but    -in .lf.general.ph    -side left

pack .lf.general.scale.en                        -in .lf.general.scale    -side bottom
pack .lf.general.scale.txt                       -in .lf.general.scale    -side left

pack .lf.onepeak.peak.sb  .lf.onepeak.peak.txt  -in .lf.onepeak.peak  -side bottom

pack .lf.onepeak.shift.sb                        -in .lf.onepeak.shift  -side bottom
pack .lf.onepeak.shift.txt  .lf.onepeak.shift.but -in .lf.onepeak.shift  -side left

pack .lf.onepeak.area.sb                        -in .lf.onepeak.area  -side bottom
pack .lf.onepeak.area.txt  .lf.onepeak.area.but -in .lf.onepeak.area  -side left

pack .lf.onepeak.gl.sb                          -in .lf.onepeak.gl    -side bottom
pack .lf.onepeak.gl.txt    .lf.onepeak.gl.but   -in .lf.onepeak.gl    -side left

pack .lf.onepeak.width.sb                        -in .lf.onepeak.width -side bottom
pack .lf.onepeak.width.txt .lf.onepeak.width.but -in .lf.onepeak.width -side left

if {[llength $argv]==1} {
  set filename [lindex $argv 0]
  set filenameroot [string range $filename 0 end-4]
  set fileextension [string range $filename end-2 end]
  readandsetup $filenameroot $fileextension
} elseif {[llength $argv]==0} {
  set filename ""
} else {
  puts "\n\nsyntax: deconv2Dxy filename"
  puts "filename   = filename of topspin-spectrum"
  exit 1
}

